/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io._
import java.nio.charset.StandardCharsets
import java.security.{DigestInputStream, MessageDigest}
import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}

import com.xebialabs.deployit.plugin.api.udm.artifact.{FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact.SCAN_PLACEHOLDERS_PROPERTY_NAME
import com.xebialabs.deployit.util.{BOM, DetectBOM, TryWith}
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.SourceArtifactEnricher._
import com.xebialabs.xldeploy.packager.io.{DevNull, StreamEntry, StreamerFactory}
import org.apache.commons.codec.binary.Hex
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.util.{Failure, Try}

object SourceArtifactEnricher {
  private[SourceArtifactEnricher] val logger: Logger = LoggerFactory.getLogger(classOf[SourceArtifactEnricher])

  private[SourceArtifactEnricher] sealed trait ScanType

  private[SourceArtifactEnricher] case object DigestOnly extends ScanType

  private[SourceArtifactEnricher] case object ProcessArchive extends ScanType

  private[SourceArtifactEnricher] case object ProcessTextFile extends ScanType

  private[SourceArtifactEnricher] case class ScanSpec(scanPlaceholders: Boolean, excludePathRegex: Option[Pattern], textFileRegex: Pattern) {
    def this(fa: SourceArtifact) = this(
      shouldScanPlaceholders(fa),
      Option(fa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(compilePattern),
      compilePattern(fa.getTextFileNamesRegex)
    )
  }

  implicit class ScanHandler(scanSpec: ScanSpec)(implicit streamerFactory: StreamerFactory) {
    def getScanType(entry: StreamEntry): ScanType = {
      if (!scanSpec.scanPlaceholders) DigestOnly
      else if (scanSpec.excludePathRegex.exists(_.matcher(entry.getPath).matches())) DigestOnly
      else if (streamerFactory.isArchive(entry.getName)) ProcessArchive
      else if (scanSpec.textFileRegex.matcher(entry.getName).matches()) ProcessTextFile
      else DigestOnly
    }
  }

  val defaultDelims = "{{ }}"
  var patternMap: Map[String, Pattern] = Map()
  val devNull = new DevNull
  val devNullWriter = new OutputStreamWriter(devNull)

  private[this] def compilePattern(regex: String): Pattern = patternMap.get(regex) match {
    case Some(compiled) => compiled
    case None =>
      val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
      patternMap += (regex -> compiled)
      compiled
  }

  private[SourceArtifactEnricher] def hasCheckSum(bda: SourceArtifact): Boolean = {
    bda.getChecksum != null && !bda.getChecksum.isEmpty
  }

  private[SourceArtifactEnricher] def getDelimiters(artifact: SourceArtifact): String = if (artifact.hasProperty("delimiters")) {
    artifact.getProperty[String]("delimiters")
  } else {
    defaultDelims
  }


  private[SourceArtifactEnricher] def shouldScanPlaceholders(artifact: SourceArtifact): Boolean = {
    !artifact.hasProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME) || artifact.getProperty[Boolean](SCAN_PLACEHOLDERS_PROPERTY_NAME)
  }
}

case class StreamFile(name: String, inputStream: InputStream)

class SourceArtifactEnricher(streamerFactory: StreamerFactory) {
  implicit val sf: StreamerFactory = streamerFactory

  def enrichArtifact(sa: SourceArtifact): Unit = {
    enrichArtifact(sa, None, devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: StreamFile): Unit = {
    enrichArtifact(sa, Option(streamFile), devNull)
  }

  def enrichArtifact(sa: SourceArtifact, streamFile: Option[StreamFile], os: OutputStream): Unit = {
    val digest: Option[MessageDigest] = if (!hasCheckSum(sa)) Some(MessageDigest.getInstance("SHA1")) else None
    val mustacher = Mustacher(getDelimiters(sa))

    sa match {
      case sa: SourceArtifact if !shouldScanPlaceholders(sa) && hasCheckSum(sa) =>
        logger.info(s"Not enriching $sa as no placeholders should be scanned and checksum is present")
      case fa: FolderArtifact =>
        logger.info(s"Going to scan artifact $fa as a folder artifact")
        scanFolderArtifact(fa, streamFile, digest, mustacher)
      case _ if streamFile.isEmpty =>
        logger.info(s"Going to scan artifact $sa")
        TryWith(sa.getFile.getInputStream) { fis =>
          scanFileOrArchiveArtifact(sa, StreamFile(sa.getFile.getName, fis), os, digest, mustacher)
        }
      case _ =>
        logger.info(s"Going to scan artifact $sa from stream")
        scanFileOrArchiveArtifact(sa, streamFile.get, os, digest, mustacher)
    }

    if (shouldScanPlaceholders(sa)) {
      sa.setPlaceholders(mustacher.placeholders.asJava)
    }
    digest.foreach(d => sa.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, Hex.encodeHexString(d.digest())))
  }

  private[this] def updateDigestWithFilename(entry: StreamEntry, digest: MessageDigest): Unit = {
    var digestPath = entry.getPath + (if (entry.isDirectory) File.separator else "")
    digestPath = digestPath.replace("\\", "/")
    logger.trace(s"scanFolderArtifact: Digest path [$digestPath]")
    digest.update(digestPath.getBytes(StandardCharsets.UTF_8))
  }

  private[this] def scanFolderArtifact(fa: FolderArtifact, streamFile: Option[StreamFile], digest: Option[MessageDigest],
                                 mustacher: Mustacher): Unit = {
    val streamer = streamFile match {
      case Some(x) => streamerFactory.streamer(x.inputStream, x.name)
      case None => streamerFactory.streamer(fa.getFile)
    }
    streamer.stream().foreach { entry =>
      digest.foreach(d => updateDigestWithFilename(entry, d))
      if (!entry.isDirectory) {
        doScan(fa, entry, mustacher, is => digest.map(new DigestInputStream(is, _)).getOrElse(is))
      }
    }
  }

  private[this] def scanFileOrArchiveArtifact(sa: SourceArtifact, streamFile: StreamFile, os: OutputStream,
                                              digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val inputStream = digest.map(new DigestInputStream(streamFile.inputStream, _)).getOrElse(streamFile.inputStream)
    if (shouldScanPlaceholders(sa)) {
      streamerFactory.streamer(inputStream, streamFile.name).stream().foreach { entry =>
        if (!entry.isDirectory) {
          doScan(sa, entry, mustacher, identity[InputStream])
        }
      }
      // Ensure that we empty the stream so that all bytes are read, testcases seem to prove this is not needed,
      // But we don't want to accidentally calculate wrong checksum.
      OverthereUtils.write(inputStream, os)
    } else {
      logger.debug(s"Artifact [$sa] has disabled placeholder scanning")
      OverthereUtils.write(inputStream, os)
    }
  }

  private[this] def doScan(sa: SourceArtifact, entry: StreamEntry, mustacher: Mustacher,
                           transform: InputStream => InputStream): Try[Unit] = {

    val scanSpec = new ScanSpec(sa)
    TryWith(entry.getInputStream) { inputStream =>
      val is = transform(inputStream)

      scanSpec.getScanType(entry) match {
        case DigestOnly =>
          logger.debug(s"$sa: Skipping ${entry.getName}")
          OverthereUtils.write(is, devNull) // Just write the bytes

        case ProcessArchive =>
          logger.debug(s"$sa: Detected archive for ${entry.getName}")
          streamerFactory.streamer(is, entry.getName).stream().foreach { e2 =>
            if (!e2.isDirectory) {
              doScan(sa, e2, mustacher, identity[InputStream])
            }
          }
          // Ensure that we empty the stream so that all bytes are read, so that we don't accidentally calculate wrong checksum.
          OverthereUtils.write(is, devNull) // Finish the archive

        case ProcessTextFile =>
          logger.debug(s"$sa: ${entry.getName} is a text file.")
          val resettableInputStream = getResettableInputStream(is)
          val reader: InputStreamReader = DetectBOM.detect(resettableInputStream) match {
            case BOM.NONE => new InputStreamReader(resettableInputStream)
            case bom: BOM => new InputStreamReader(resettableInputStream, bom.getCharset)
          }
          TryWith(mustacher.newReader(reader)) { r =>
            OverthereUtils.write(r, devNullWriter) // Scan the file for placeholders using the Mustacher
          }
      }
    }
  }

  private def getResettableInputStream(is: InputStream): InputStream = {
    Try(is.reset()) match {
      case Failure(_) => new BufferedInputStream(is)
      case _ => is
    }
  }
}
