/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io.Reader

import com.xebialabs.xldeploy.packager.Delims.{Full, Match, No, Partial}
import com.xebialabs.xldeploy.packager.Mustacher._
import org.slf4j.{Logger, LoggerFactory}

class MustacheScanningReader(mustacher: Mustacher, wrapped: Reader) extends Reader {
  override def read(chars: Array[Char], off: Int, len: Int): Int = {
    val readCount = wrapped.read(chars, off, len)
    if (readCount > 0) {
      mustacher.handle(chars, off, readCount)
    }
    readCount
  }

  override def close(): Unit = {
    wrapped.close()
    mustacher.resetState()
  }
}

object Delims {
  sealed trait Match
  case object No extends Match
  case object Partial extends Match
  case object Full extends Match
}

class Delims(delims: String) {
  require((delims.length == 5 && delims(2) == ' ') || (delims.length == 3 && delims(1) == ' '), "Wrong format of delimiters provided, should be '12 34' or '1 2', where 1, 2, 3 and 4 delimiter characters")
  val Array(start, end) = delims.split(' ')

  def matchesStart(c1: Char): Match = {
    matches(c1, '\u0000', start)
  }

  def matchesStart(c1: Char, c2: Char): Match = {
    matches(c1, c2, start)
  }

  def matchesEnd(c1: Char): Match = {
    matches(c1, '\u0000', end)
  }

  def matchesEnd(c1: Char, c2: Char): Match = {
    matches(c1, c2, end)
  }

  private[this] def matches(c1: Char, c2: Char, delim: String): Match = {
    if (c1 != delim(0)) {
      No
    } else if (delim.length == 1) {
      Full
    } else if (delim.length == 2 && c2 == delim(1)) {
      Full
    } else {
      Partial
    }

  }
}

object Mustacher {
  val logger: Logger = LoggerFactory.getLogger(classOf[Mustacher])

  sealed trait State
  case class MatchingTag(c: Array[Char]) extends State
  case object MatchStartTag extends State
  case class MatchNestedStart(c: Array[Char]) extends State
  case class MatchEndTag(tag: Array[Char]) extends State
  case object Text extends State

  def apply(delims: String): Mustacher = new Mustacher(new Delims(delims))
}

class Mustacher(delims: Delims) {
  var placeholders: Set[String] = Set()
  var state: Mustacher.State = Mustacher.Text
  var lookBehind: Char = _
  val invalids = Set('\r', '\n')


  def newReader(reader: Reader): MustacheScanningReader = new MustacheScanningReader(this, reader)

  def handle(contents: Array[Char], offset: Int, len: Int): Unit = {
    for (i <- 0 until len) {
      handle(contents(offset + i))
    }
  }

  def handle(c: Char): Unit = {
    state match {
      case Text if delims.matchesStart(c) == Partial => // 2-char start delimiter
        state = Mustacher.MatchStartTag
      case Text if delims.matchesStart(c) == Full => // 1-char start delimiter
        state = MatchingTag(Array())
      case Text => // Ok
      case MatchStartTag if delims.matchesStart(lookBehind, c) == Full => // 2-char start delimiter
        state = MatchingTag(Array())
      case MatchStartTag => // Not matched delimiter, continue in text mode
        state = Text
      case MatchingTag(tag) if delims.matchesEnd(c) == Full => // 1-char end delimiter
        val stringTag = tag.mkString.trim
        logger.debug(s"Found placeholder: [$stringTag]")
        placeholders += stringTag
        state = Text
      case MatchingTag(cs) if delims.matchesEnd(c) == Partial => // 2-char end delimiter
        state = MatchEndTag(cs)
      case MatchingTag(cs) if delims.matchesStart(c) == Partial => // 2-char nested delimiter start
        state = MatchNestedStart(cs)
      case MatchingTag(_) if delims.matchesStart(c) == Full => // 2-char nested delimiter start
        state = MatchingTag(Array()) // Throw away currently matched tag
      case MatchingTag(cs) if invalids.contains(c) =>
        // No support for multiline placeholders, stop processing
        logger.debug(s"Found and skipping invalid placeholder with linefeed and/or carriage return character: [${cs.mkString.trim}]")
        state = Text
      case MatchingTag(cs) =>
        state = MatchingTag(cs :+ c)
      case MatchNestedStart(_) if delims.matchesStart(lookBehind, c) == Full => // 2-char nested delimiter start
        state = MatchingTag(Array()) // Throw away currently matched tag
      case MatchNestedStart(cs) => // Not matched full nested start-delimiter, continue tag-matching
        state = MatchingTag(cs :+ c)
      case MatchEndTag(tag) if delims.matchesEnd(lookBehind, c) == Full => // 2-char end delimiter
        val stringTag = tag.mkString.trim
        logger.debug(s"Found placeholder: [$stringTag]")
        placeholders += stringTag
        state = Text
       case MatchEndTag(tag) => // Not matched full end-delimiter, continue tag-matching
        state = MatchingTag(tag :+ lookBehind :+ c)
    }
    lookBehind = c
  }

  def resetState(): Unit = {
    state = Text
  }
}