/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io.{Closeable, InputStream}

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xldeploy.packager.PackagerConfig
import org.apache.commons.compress.archivers.jar.JarArchiveInputStream
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveInputStream}
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.util.Try

sealed trait StreamEntry {
  def getName: String

  def getPath: String

  def getInputStream: InputStream

  def isDirectory: Boolean
}

case class ArchivedEntry(ze: ArchiveEntry, stream: ArchiveInputStream) extends StreamEntry {
  override def getName: String = ze.getName.split('/').last

  override def getPath: String = if (isDirectory) {
    ze.getName.substring(0, ze.getName.length - 1)
  } else {
    ze.getName
  }

  override def getInputStream: InputStream = new InputStream {

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def read(): Int = stream.read()

    override def close(): Unit = {}
  }

  override def isDirectory: Boolean = ze.isDirectory
}

case class DirectoryEntry(file: OverthereFile, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = throw new UnsupportedOperationException("directory does not have getInputStream")

  override def isDirectory: Boolean = true
}

case class FileEntry(file: OverthereFile, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = file.getInputStream

  override def isDirectory: Boolean = false
}

case class FileStreamEntry(stream: InputStream, name: String) extends StreamEntry {
  override def getName: String = name

  override def getPath: String = name

  override def getInputStream: InputStream = new InputStream {
    override def read(): Int = stream.read()

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def markSupported(): Boolean = stream.markSupported()

    override def available(): Int = stream.available()

    override def skip(n: Long): Long = stream.skip(n)

    override def reset(): Unit = stream.reset()

    override def close(): Unit = {} // Do not close ourselves, this object does not own the stream

    override def mark(readlimit: Int): Unit = stream.mark(readlimit)
  }

  override def isDirectory = false
}

sealed trait Streamer {
  def stream(): Stream[StreamEntry]
}

object StreamerFactory {
  val logger: Logger = LoggerFactory.getLogger(classOf[StreamerFactory])
  def defaultMappings(): StreamerFactory = forConfig(ConfigFactory.defaultReference())
  def forConfig(config: Config) = new StreamerFactory(new PackagerConfig(config).archiveExtensionMappings)
}

class StreamerFactory(archiveMappings: Map[String, String]) {
  private[this] def getExtension(name: String): String = {
    val i = name.indexOf('.')
    if (i > 0) name.substring(i + 1) else ""
  }

  def getArchiveStream(name: String, is: InputStream): ArchiveInputStream = {
    val origExt = getExtension(name)
    val ext = archiveMappings(origExt)
    StreamerFactory.logger.debug(s"Detected mapped archive extension $origExt->$ext for $name")
    ext match {
      case "jar" => new JarArchiveInputStream(is)
      case "zip" => new ZipArchiveInputStream(is)
      case "tar" => new TarArchiveInputStream(is)
      case "tar.gz" => new TarArchiveInputStream(new GzipCompressorInputStream(is))
      case "tar.bz2" => new TarArchiveInputStream(new BZip2CompressorInputStream(is))
      case _ => throw UnsupportedArchiveExtensionException(s"$name with extension $origExt mapped to $ext, which is not a supported archive")
    }
  }

  def isArchive(name: String): Boolean = {
    val ext = getExtension(name)
    StreamerFactory.logger.debug(s"Detecting whether $name is an archive (Found extension $ext)")
    archiveMappings.contains(ext)
  }

  def streamer(is: InputStream, name: String): Streamer = {
    if (isArchive(name)) {
      new ArchiveStreamStreamer(is, name, this)
    } else {
      new FileStreamStreamer(is, name)
    }
  }

  def streamer(file: OverthereFile): Streamer = {
    if (file.isDirectory) {
      new DirectoryStreamer(file, this)
    } else if (isArchive(file.getName)) {
      new ArchiveStreamer(file, this)
    } else {
      new FileStreamer(file)
    }
  }
}

class DirectoryStreamer(dir: OverthereFile, sf: StreamerFactory) extends Streamer {
  override def stream(): Stream[StreamEntry] = {
    def fileStream(dir: OverthereFile, base: String): Stream[StreamEntry] =
      if (dir.isDirectory)
        Option(dir.listFiles)
          .map(_.asScala.toList.sortBy(_.getPath).toStream.flatMap(file =>
            if (file.isDirectory) {
              DirectoryEntry(file, base) #:: fileStream(file, base)
            } else {
              FileEntry(file, base) #:: fileStream(file, base)
            }))
          .getOrElse(Stream.empty)
      else Stream.empty

    val base = dir.getPath + "/"
    fileStream(dir, base)
  }
}

class FileStreamer(file: OverthereFile) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileEntry(file, file.getParentFile.getPath + "/") #:: Stream.empty[StreamEntry]
}

class FileStreamStreamer(is: InputStream, name: String) extends Streamer {
  override def stream(): Stream[StreamEntry] = FileStreamEntry(is, name) #:: Stream.empty[StreamEntry]
}

trait ArchiveEntryStreamer extends Streamer {
  def streamerFactory: StreamerFactory

  def nextEntry(s: ArchiveInputStream, toClose: Set[Closeable]): Stream[StreamEntry] = {
    Option(s.getNextEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        Stream.empty
      case Some(e) => ArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

class ArchiveStreamer(file: OverthereFile, val streamerFactory: StreamerFactory) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val is = file.getInputStream
    val archiveStream = streamerFactory.getArchiveStream(file.getName, is)
    nextEntry(archiveStream, Set(is, archiveStream))
  }
}

class ArchiveStreamStreamer(is: InputStream, name: String, val streamerFactory: StreamerFactory) extends ArchiveEntryStreamer {
  override def stream(): Stream[StreamEntry] = {
    val archiveStream = streamerFactory.getArchiveStream(name, is)
    nextEntry(archiveStream, Set())
  }
}

case class UnsupportedArchiveExtensionException(str: String) extends RuntimeException(str)