/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io.InputStream

import org.apache.commons.compress.archivers.ArchiveStreamFactory
import org.apache.commons.compress.compressors.CompressorStreamFactory
import org.slf4j.LoggerFactory

import scala.util.{Failure, Success, Try}

object SupportedArchiveExtensions {
  private val logger = LoggerFactory.getLogger(SupportedArchiveExtensions.getClass)

  val ZIP = "zip"
  val JAR = "jar"
  val TAR = "tar"
  val TARGZ = "tar.gz"
  val TARBZ2 = "tar.bz2"

  def archiveTypeMatchesExtension(is: InputStream, expectedType: String, name: String): Boolean = {
    val detectedArchiveType: Option[String] = Try(ArchiveStreamFactory.detect(is)) match {
      case Success(archiveType) => Option(archiveType)
      case Failure(_) => Try(CompressorStreamFactory.detect(is)) match {
        case Success(archiveType) => Option(archiveType)
        case Failure(_) => Option.empty[String]
      }
    }

    (detectedArchiveType, expectedType) match {
      case (Some(CompressorStreamFactory.BZIP2), TARBZ2) => true
      case (Some(CompressorStreamFactory.GZIP), TARGZ) => true
      case (Some(ArchiveStreamFactory.TAR), TAR) => true
      case (Some(ArchiveStreamFactory.ZIP), ZIP | JAR) => true
      case (Some(detected), _) =>
        logger.warn(s"Found archive ($name) with mapped type .$expectedType, but the actual type is .$detected - skipping")
        false
      case (
        None, TARBZ2 | TARGZ | TAR | ZIP | JAR) =>
        logger.warn(s"Found file ($name) with mapped type .$expectedType, but the file is not a valid archive type")
        false
      case (None, _) =>
        throw UnsupportedArchiveExtensionException(s"$name with extension $expectedType is not a supported archive")
    }
  }

}
