/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io._
import java.util.{Arrays => JArrays, Enumeration => JEnumeration}

import org.apache.commons.compress.archivers.ArchiveInputStream
import org.apache.commons.compress.archivers.jar.JarArchiveInputStream
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.zip._

import scala.jdk.CollectionConverters._
import scala.util.Try

sealed trait Streamer {
  def stream(): LazyList[StreamEntry]
}

class DirectoryStreamer(dir: File) extends Streamer {

  private[this] def fileStream(dir: File, base: String): LazyList[StreamEntry] = {
    if (!dir.isDirectory) { return LazyList.empty }

    Option(dir.listFiles)
      .map(a => JArrays.asList(a: _*).asScala.sortBy(_.getPath).to(LazyList).flatMap(file =>
        if (file.isDirectory) {
          DirectoryEntry(file, base) #:: fileStream(file, base)
        } else {
          FileEntry(file, base) #:: fileStream(file, base)
        }))
      .getOrElse(LazyList.empty)
  }

  override def stream(): LazyList[StreamEntry] = fileStream(dir, dir.getPath + "/")
}

class EmptyStreamer() extends Streamer {
  override def stream(): LazyList[StreamEntry] = LazyList.empty[StreamEntry]
}

class FileStreamer(file: File) extends Streamer {
  override def stream(): LazyList[StreamEntry] = FileEntry(file, file.getParentFile.getPath + "/") #:: LazyList.empty[StreamEntry]
}

class FileStreamStreamer(is: InputStream, name: String) extends Streamer {
  override def stream(): LazyList[StreamEntry] = FileStreamEntry(is, name) #:: LazyList.empty[StreamEntry]
}

trait ArchiveEntryStreamer extends Streamer {
  def nextEntry(s: ArchiveInputStream, toClose: Set[Closeable]): LazyList[StreamEntry] = {
    Option(s.getNextEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        LazyList.empty
      case Some(e) => ArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait ZipEntryStreamer extends Streamer {
  def nextEntry(s: ZipArchiveInputStream, toClose: Set[Closeable]): LazyList[StreamEntry] = {
    Option(s.getNextZipEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        LazyList.empty
      case Some(e) => ZipArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait ZipFileEntryStreamer extends Streamer {
  def nextEntry(zf: ZipFile, zfe: JEnumeration[ZipArchiveEntry], toClose: Set[Closeable]): LazyList[StreamEntry] = {
    if (zfe.hasMoreElements) {
      val e = zfe.nextElement()
      ZipArchivedZipFileEntry(e, zf) #:: nextEntry(zf, zfe, toClose)
    } else {
      toClose.foreach(c => Try(c.close()))
      LazyList.empty
    }
  }
}

trait JarEntryStreamer extends Streamer {
  def nextEntry(s: JarArchiveInputStream, toClose: Set[Closeable]): LazyList[StreamEntry] = {
    Option(s.getNextJarEntry) match {
      case None =>
        toClose.foreach(c => Try(c.close()))
        LazyList.empty
      case Some(e) => JarArchivedEntry(e, s) #:: nextEntry(s, toClose)
    }
  }
}

trait JarFileEntryStreamer extends Streamer {
  def nextEntry(zf: ZipFile, zfe: JEnumeration[ZipArchiveEntry], toClose: Set[Closeable]): LazyList[StreamEntry] = {
    if (zfe.hasMoreElements) {
      val e = zfe.nextElement()
      JarArchivedZipFileEntry(e, zf) #:: nextEntry(zf, zfe, toClose)
    } else {
      toClose.foreach(c => Try(c.close()))
      LazyList.empty
    }
  }
}

class ZipFileArchiveStreamer(file: File) extends ZipFileEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val zf = new ZipFile(file)
    nextEntry(zf, zf.getEntriesInPhysicalOrder, Set(zf))
  }
}

class ZipArchiveStreamStreamer(is: InputStream) extends ZipEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val archiveStream = new ZipArchiveInputStream(is)
    nextEntry(archiveStream, Set.empty)
  }
}

class JarFileArchiveStreamer(file: File) extends JarFileEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val zf = new ZipFile(file)
    nextEntry(zf, zf.getEntriesInPhysicalOrder, Set(zf))
  }
}

class JarArchiveStreamStreamer(is: InputStream) extends JarEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val archiveStream = new JarArchiveInputStream(is)
    nextEntry(archiveStream, Set.empty)
  }
}

// it's a +de+compressor since it's an InputStream
class ArchiveStreamer(file: File, val decompressor: InputStream => InputStream) extends ArchiveEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val is = new FileInputStream(file)
    val decompressedStream = decompressor(is)
    val archiveStream = new TarArchiveInputStream(decompressedStream)
    nextEntry(archiveStream, Set(is, decompressedStream, archiveStream))
  }
}

// it's a +de+compressor since it's an InputStream
class ArchiveStreamStreamer(is: InputStream, val decompressor: InputStream => InputStream) extends ArchiveEntryStreamer {
  override def stream(): LazyList[StreamEntry] = {
    val decompressedStream = decompressor(is)
    val archiveStream = new TarArchiveInputStream(decompressedStream)
    nextEntry(archiveStream, Set.empty)
  }
}

final case class UnsupportedArchiveExtensionException(str: String) extends RuntimeException(str)
