/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io.{BufferedInputStream, File, FileInputStream, InputStream}

import org.apache.commons.compress.archivers.ArchiveEntry
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry

sealed trait StreamEntry {
  def getName: String

  def getPath: String

  def getInputStream: InputStream

  def isDirectory: Boolean
}

trait XLArchiveEntry extends StreamEntry {
  def path: String

  def stream: InputStream

  def ze: ArchiveEntry

  override def getName: String = path.split('/').last

  override def getPath: String = if (isDirectory) {
    path.substring(0, path.length - 1)
  } else {
    path
  }

  override def getInputStream: InputStream = new BufferedInputStream(new InputStream {

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def read(): Int = stream.read()

    override def close(): Unit = {}
  })
}

final case class ArchivedEntry(ze: ArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

final case class ZipArchivedEntry(ze: ZipArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

final case class JarArchivedEntry(ze: ZipArchiveEntry, stream: InputStream) extends XLArchiveEntry {
  override val path: String = ze.getName

  override def isDirectory: Boolean = ze.isDirectory
}

final case class DirectoryEntry(file: File, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = throw new UnsupportedOperationException("directory does not have getInputStream")

  override def isDirectory: Boolean = true
}

final case class FileEntry(file: File, base: String) extends StreamEntry {
  override def getName: String = file.getName

  override def getPath: String = file.getPath.substring(base.length)

  override def getInputStream: InputStream = new FileInputStream(file)

  override def isDirectory: Boolean = false
}

final case class FileStreamEntry(stream: InputStream, name: String) extends StreamEntry {
  override def getName: String = name

  override def getPath: String = name

  override def getInputStream: InputStream = new InputStream {
    override def read(): Int = stream.read()

    override def read(b: Array[Byte]): Int = stream.read(b)

    override def read(b: Array[Byte], off: Int, len: Int): Int = stream.read(b, off, len)

    override def markSupported(): Boolean = stream.markSupported()

    override def available(): Int = stream.available()

    override def skip(n: Long): Long = stream.skip(n)

    override def reset(): Unit = stream.reset()

    override def close(): Unit = {} // Do not close ourselves, this object does not own the stream

    override def mark(readlimit: Int): Unit = stream.mark(readlimit)
  }

  override def isDirectory = false
}


