/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.placeholders

import java.util.regex.Pattern
import java.util.regex.Pattern.{CASE_INSENSITIVE, COMMENTS}

import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.xldeploy.packager.io.{StreamEntry, StreamerFactory}
import PlaceholdersUtil.SourceArtifactUtil
import com.xebialabs.xldeploy.packager.MustacherReplacer

object ScanSpec {

  var patternMap: Map[String, Pattern] = Map()

  private def compilePattern(regex: String): Pattern = patternMap.get(regex) match {
    case Some(compiled) => compiled
    case None =>
      val compiled = Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE)
      patternMap += (regex -> compiled)
      compiled
  }

}

private[placeholders] final case class ScanSpec(scanPlaceholders: Boolean,
                                                excludePathRegex: Option[Pattern],
                                                textFileRegex: Pattern,
                                                mustacher: MustacherReplacer) {

  def this(sa: SourceArtifact) = this(
    sa.shouldScanPlaceholders(),
    Option(sa.getExcludeFileNamesRegex).filterNot(_.isEmpty).map(ScanSpec.compilePattern),
    ScanSpec.compilePattern(sa.getTextFileNamesRegex),
    MustacherReplacer(sa.mustacheDelimiters())
  )

  def getProcessingType(entry: StreamEntry)(implicit streamerFactory: StreamerFactory): ProcessingType = {
    if (!scanPlaceholders) DigestOnly
    else if (excludePathRegex.exists(_.matcher(entry.getPath).matches())) DigestOnly
    else if (streamerFactory.hasArchiveExtension(entry.getName)) ProcessArchive
    else if (textFileRegex.matcher(entry.getName).matches()) ProcessTextFile
    else DigestOnly
  }
}
