/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager

import java.io.Reader
import com.xebialabs.xldeploy.packager.Delims._
import com.xebialabs.xldeploy.packager.MustacheReplacingReader.END_OF_STREAM_VALUE
import com.xebialabs.xldeploy.packager.MustacherReplacer._
import grizzled.slf4j.Logger
import org.slf4j.LoggerFactory

import java.util
import scala.annotation.tailrec

object MustacheReplacingReader {
  private val END_OF_STREAM_VALUE: Int = -1
}

class MustacheReplacingReader(mustacher: MustacherReplacer, wrapped: Reader, placeholders: Map[String, String]) extends Reader {
  import com.xebialabs.xldeploy.packager.MustacherReplacer._

  override def read(chars: Array[Char]): Int = read(chars, 0, chars.length)

  override def read(chars: Array[Char], off: Int, len: Int): Int = {
    var charsRead: Int = 0
    var i: Int = 0
    while ( {
      i < len
    }) {
      val nextChar: Int = read()
      if (nextChar == END_OF_STREAM_VALUE) {
        if (charsRead == 0) {
          charsRead = END_OF_STREAM_VALUE
        }
        return charsRead
      }
      charsRead = i + 1
      chars(off + i) = nextChar.toChar

      i += 1
    }
    charsRead
  }

  @tailrec
  override final def read(): Int = {
    mustacher.handleReplace(wrapped, placeholders) match {
      case Letter(value) => value
      case EndOfStream => END_OF_STREAM_VALUE
      case Nothing => read()
    }
  }

  override def close(): Unit = {
    wrapped.close()
    mustacher.resetState()
  }
}

object MustacherReplacer extends MustacheReader {

  private[MustacherReplacer] val logger = new Logger(LoggerFactory.getLogger(classOf[MustacherReplacer]))

  private val invalids = Set('\r', '\n')

  def apply(delims: String): MustacherReplacer = new MustacherReplacer(new Delims(delims))

  def appended3(a1: Array[Char], a2: Array[Char], a3: Array[Char]): Array[Char] = {
    val dest = util.Arrays.copyOf(a1, a1.length + a2.length + a3.length)
    Array.copy(a2, 0, dest, a1.length, a2.length)
    Array.copy(a3, 0, dest, a1.length + a2.length, a3.length)
    dest
  }

  def appended2AndChar(a1: Array[Char], a2: Array[Char], c3: Char): Array[Char] = {
    val dest = util.Arrays.copyOf(a1, a1.length + a2.length + 1)
    Array.copy(a2, 0, dest, a1.length, a2.length)
    dest(dest.length - 1) = c3
    dest
  }

  sealed trait BufferState
  final case class Letter(c: Int) extends BufferState
  case object Nothing extends BufferState
  case object EndOfStream extends BufferState
}

class MustacherReplacer(delims: Delims) {
  var placeholders: Set[String] = Set.empty
  var state: MustacherReplacer.State = MustacherReplacer.Text
  private var lookBehind: Char = _

  var replacementBuffer: Array[Char] = Array.empty
  var replacementBufferPosition = 0

  def newReader(reader: Reader, placeholders: Map[String, String]): MustacheReplacingReader = new MustacheReplacingReader(this, reader, placeholders)

  def handleReplace(reader: Reader, replacements: Map[String, String]): BufferState = {
    if (replacementBuffer.nonEmpty) {
      readFromReplacementBuffer()
    } else {
      readFromOriginalBuffer(reader, replacements)
    }
  }

  def resetState(): Unit = {
    state = Text
  }

  def resolve(tag: String, placeholders: Map[String, String]): Array[Char] = {
    if (!placeholders.contains(tag)) {
      logger.error(s"Could not find a replacement for [$tag] placeholder.")
    }
    placeholders.get(tag) match {
      case Some(rb: String) if rb == IGNORE_PLACEHOLDER => appended3(delims.start, tag.toCharArray, delims.end)
      case Some(rb: String) if rb == EMPTY_PLACEHOLDER => Array.empty
      case Some(rb: String) => rb.toCharArray
      case _ => appended3(delims.start, tag.toCharArray, delims.end)
    }
  }

  private def readFromReplacementBuffer(): BufferState = {
    replacementBuffer match {
      case replacement if replacement.nonEmpty && replacementBufferPosition < replacement.length =>
        val result = replacement(replacementBufferPosition)
        replacementBufferPosition += 1
        Letter(result)
      case _ =>
        replacementBuffer = Array.empty
        replacementBufferPosition = 0
        Nothing
    }
  }

  private def readFromOriginalBuffer(reader: Reader, replacements: Map[String, String]): BufferState  = {
    val data: Int = reader.read()
    if (data != -1) {
      val c = data.toChar
      val result = state match {
        case Text if delims.matchesStart(c) == Partial => // 2-char start delimiter
          state = MatchStartTag
          Nothing
        case Text if delims.matchesStart(c) == Full => // 1-char start delimiter
          state = matchingTagEmptyArray
          Nothing
        case Text => // Ok
          Letter(c.toInt)
        case MatchStartTag if delims.matchesStart(lookBehind, c) == Full => // 2-char start delimiter
          state = matchingTagEmptyArray
          Nothing
        case MatchStartTag => // Not matched delimiter, continue in text mode
          replacementBuffer = Array(delims.start(0), c)
          state = Text
          Nothing
        case MatchingTag(tag) if delims.matchesEnd(c) == Full => // 1-char end delimiter
          val stringTag = tag.mkString.trim
          logger.debug(s"Found placeholder: [$stringTag]")
          placeholders += stringTag
          replacementBuffer = resolve(stringTag, replacements)
          logger.debug(s"It should be replaced with: [${replacementBuffer.mkString}]")
          state = Text
          Nothing
        case MatchingTag(cs) if delims.matchesEnd(c) == Partial => // 2-char end delimiter
          state = MatchEndTag(cs)
          Nothing
        case MatchingTag(cs) if delims.matchesStart(c) == Partial => // 2-char nested delimiter start with partial match for previously found
          state = MatchNestedStart(cs)
          Nothing
        case MatchingTag(cs) if delims.matchesStart(c) == Full => // 2-char nested delimiter start with full match for previously found and possibly some characters scanned
          replacementBuffer = delims.start ++ cs
          state = matchingTagEmptyArray // Throw away currently matched tag
          Nothing
        case MatchingTag(cs) if invalids.contains(c) =>
          // No support for multiline placeholders, stop processing
          logger.debug(s"Found and skipping invalid placeholder with linefeed and/or carriage return character: [${cs.mkString.trim}]")
          replacementBuffer = appended2AndChar(delims.start, cs, c)
          state = Text
          Nothing
        case MatchingTag(cs) =>
          state = MatchingTag(cs :+ c)
          Nothing
        case MatchNestedStart(cs) if delims.matchesStart(lookBehind, c) == Full => // 2-char nested delimiter start
          val prefix = if (cs.isEmpty) Array(delims.start(0)) else delims.start
          replacementBuffer = prefix ++ cs
          state = matchingTagEmptyArray // Throw away currently matched tag
          Nothing
        case MatchNestedStart(cs) if cs.isEmpty => // Not matched full nested start-delimiter with empty placeholder tag, continue tag-matching
          replacementBuffer = Array(delims.start(0))
          state = MatchingTag(Array[Char](1) :+ c)
          Nothing
        case MatchNestedStart(cs) if !cs.isEmpty => // Not matched full nested start-delimiter, continue tag-matching
          val chars = cs :+ delims.start(0)
          state = MatchingTag(chars :+ c)
          Nothing
        case MatchEndTag(tag) if delims.matchesEnd(lookBehind, c) == Full => // 2-char end delimiter
          val stringTag = tag.mkString.trim
          logger.debug(s"Found placeholder: [$stringTag]")
          placeholders += stringTag
          val replaced = resolve(stringTag, replacements)
          replacementBuffer = replaced
          logger.debug(s"It should be replaced with: [${replaced.mkString}]")
          state = Text
          Nothing
        case MatchEndTag(tag) => // Not matched full end-delimiter, continue tag-matching
          state = MatchingTag(tag :+ lookBehind :+ c)
          Nothing
      }
      lookBehind = c
      result
    } else state match {
      case tag: MatchingTag =>
        replacementBuffer = delims.start ++ tag.c
        state = Text
        Nothing
      case tag: MatchNestedStart =>
        replacementBuffer = appended2AndChar(delims.start, tag.c, delims.start(0))
        state = Text
        Nothing
      case tag: MatchEndTag =>
        replacementBuffer = appended2AndChar(delims.start, tag.tag, delims.end(0))
        state = Text
        Nothing
      case _ =>
        state = Text
        EndOfStream
    }
  }
}
