/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io._
import java.nio.charset.Charset
import java.nio.charset.Charset.defaultCharset
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.util.{BOM, DetectBOM, TryWith}
import grizzled.slf4j.Logging
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveOutputStream}
import org.apache.commons.io.IOUtils

import scala.collection.JavaConverters._
import scala.util.{Failure, Success, Try}

object ArtifactIOUtils extends Logging {
  def getResettableInputStream(is: InputStream): InputStream =
    Try(is.reset()) match {
      case Failure(_) if !is.isInstanceOf[BufferedInputStream] => new BufferedInputStream(is)
      case Success(_) if is.isInstanceOf[TarArchiveInputStream] => new BufferedInputStream(is) // it does not reset correctly
      case _ => is
    }


  def detectCharset(sa: SourceArtifact, is: InputStream, entry: StreamEntry): Option[Charset] = {

    val charset: Option[Charset] = DetectBOM.detect(is) match {
      case BOM.NONE => getSpecifiedCharset(sa, entry)
      case bom: BOM => Option(bom.getCharset)
    }

    logger.debug(s"Entry ${entry.getName} has encoding: ${charset.getOrElse("Unknown")}")
    charset
  }

  def readWithCharset(is: InputStream, charset: Option[Charset]): Reader =
    charset.map(cs => new InputStreamReader(is, cs)).getOrElse(new InputStreamReader(is))

  def withOpenArchiveEntry(os: ArchiveOutputStream, ae: ArchiveEntry, writeEntryContents: () => Unit): Unit = {
    os.putArchiveEntry(ae)
    writeEntryContents()
    os.closeArchiveEntry()
  }

  def withArchiveOutputStream(file: File, block: ArchiveOutputStream => Unit)(implicit streamerFactory: StreamerFactory)
  : Unit = {
    TryWith(streamerFactory.getArchiveOutputStream(file)) { os =>
      block(os)
      os.finish()
    }
  }

  def copyChars(wrappedReader: Reader, os: OutputStream, optCharset: Option[Charset], buffer: Array[Char]): Unit = {
    val out = new OutputStreamWriter(os, optCharset.getOrElse(defaultCharset()))
    IOUtils.copyLarge(wrappedReader, out, buffer)
    out.flush()
  }

  def copyChars(reader: Reader, writer: Writer, buffer: Array[Char]): Unit = IOUtils.copyLarge(reader, writer, buffer)

  def copyBytes(is: InputStream, os: OutputStream, buffer: Array[Byte]): Unit = IOUtils.copyLarge(is, os, buffer)

  def getSpecifiedCharset(sa: SourceArtifact, entry: StreamEntry): Option[Charset] = {
    sa.getFileEncodings.asScala
      .find({ case (pathRegex, _) => entry.getPath.matches(pathRegex)})
      .map({case (pathRegex, charset) =>
        logger.debug("Relative path [{}] matched regex [{}], using charset [{}]", entry.getPath, pathRegex, charset)
        Charset.forName(charset)
      })
  }
}
