/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io._

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.util.TryWith
import com.xebialabs.xldeploy.packager.PackagerConfig
import com.xebialabs.xldeploy.packager.io.StreamerFactory.logger
import com.xebialabs.xldeploy.packager.io.SupportedArchiveExtensions._
import org.apache.commons.compress.archivers.ArchiveOutputStream
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream
import org.apache.commons.compress.compressors.bzip2.{BZip2CompressorInputStream, BZip2CompressorOutputStream}
import org.apache.commons.compress.compressors.gzip.{GzipCompressorInputStream, GzipCompressorOutputStream}
import org.slf4j.{Logger, LoggerFactory}

object StreamerFactory {
  val logger: Logger = LoggerFactory.getLogger(classOf[StreamerFactory])
  def defaultMappings(): StreamerFactory = forConfig(ConfigFactory.defaultReference())

  def forConfig(config: Config) = new StreamerFactory(new PackagerConfig(config).archiveExtensionMappings)
}

class StreamerFactory(archiveMappings: Map[String, String]) {
  // used for replacement
  private[this] def getArchiveEntryStreamer(file: File): Streamer = {
    val mappedExt = getArchiveType(file.getName)

    val extOK = TryWith(ArtifactIOUtils.getResettableInputStream(new FileInputStream(file))) { resettableIs =>
      archiveTypeMatchesExtension(resettableIs, mappedExt, file.getName)
    }.get

    mappedExt match {
      case _ if !extOK => new EmptyStreamer
      case ZIP => new ZipFileArchiveStreamer(file)
      case JAR => new JarFileArchiveStreamer(file)
      case TAR => new ArchiveStreamer(file, is => is)
      case TARGZ => new ArchiveStreamer(file, is => new GzipCompressorInputStream(is))
      case TARBZ2 => new ArchiveStreamer(file, is => new BZip2CompressorInputStream(is))
    }
  }

  // used for scanning
  private[io] def getArchiveEntryStreamStreamer(is: InputStream, name: String): Streamer = {
    val mappedExt = getArchiveType(name)

    val resettableIs = ArtifactIOUtils.getResettableInputStream(is)
    val extOK = archiveTypeMatchesExtension(resettableIs, mappedExt, name)

    mappedExt match {
      case _ if !extOK => new EmptyStreamer
      case ZIP => new ZipArchiveStreamStreamer(resettableIs)
      case JAR => new JarArchiveStreamStreamer(resettableIs)
      case TAR => new ArchiveStreamStreamer(resettableIs, is => is)
      case TARGZ => new ArchiveStreamStreamer(resettableIs, is => new GzipCompressorInputStream(is))
      case TARBZ2 => new ArchiveStreamStreamer(resettableIs, is => new BZip2CompressorInputStream(is))
    }
  }

  def getArchiveOutputStream(target: File): ArchiveOutputStream = {
    getArchiveType(target.getName) match {
      case ZIP => new ZipArchiveOutputStream(target)
      case JAR => new ZipArchiveOutputStream(target) // JarArchiveOutputStream doesn't have this constructor. It is required for correct function.
      case TAR => getTarArchiveOutputStream(target, os => os)
      case TARGZ => getTarArchiveOutputStream(target, os => new GzipCompressorOutputStream(os))
      case TARBZ2 => getTarArchiveOutputStream(target, os => new BZip2CompressorOutputStream(os))

      case mappedExt =>
        val originExt = archiveMappings.find(p = am => target.getName.endsWith('.' + am._1)).get._1
        throw UnsupportedArchiveExtensionException(
          s"${target.getName} with extension $originExt -> $mappedExt is not a supported archive output type")
    }
  }

  private[this] def getTarArchiveOutputStream(target: File, compressor: OutputStream => OutputStream): TarArchiveOutputStream = {
    val wrapped = compressor(new FileOutputStream(target)) // TODO will these be closed properly?
    val os = new TarArchiveOutputStream(wrapped)
    os.setLongFileMode(TarArchiveOutputStream.LONGFILE_POSIX)
    os.setBigNumberMode(TarArchiveOutputStream.BIGNUMBER_POSIX)
    os
  }

  def hasArchiveExtension(name: String): Boolean = archiveMappings.exists(am => name.endsWith('.' + am._1))

  def streamer(is: InputStream, name: String): Streamer = {
    if (hasArchiveExtension(name)) {
      getArchiveEntryStreamStreamer(is, name)
    } else {
      new FileStreamStreamer(is, name)
    }
  }

  def streamer(file: File): Streamer = {
    if (file.isDirectory) {
      new DirectoryStreamer(file)
    } else if (hasArchiveExtension(file.getName)) {
      getArchiveEntryStreamer(file)
    } else {
      new FileStreamer(file)
    }
  }

  private[this] def getArchiveType(name: String): String =
    archiveMappings.find(am => name.endsWith('.' + am._1)) match {
      case None =>
        throw UnsupportedArchiveExtensionException(s"$name is not a supported archive")
      case Some((originExt, mappedExt)) =>
        logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for $name")
        mappedExt
    }

}

