package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.booter.local.utils.XmlUtils.Closure;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.reflect.TypeVerification;
import com.xebialabs.deployit.plugin.api.reflect.VerificationContext;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployable;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployedContainer;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import com.xebialabs.deployit.plugin.api.udm.Prefix;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.booter.local.utils.ClassUtils.getActualTypeArguments;
import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.searchField;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.childByName;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.childrenByName;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.forEach;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getOptionalBooleanAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getOptionalStringAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getOptionalTypeAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredStringAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredTypeAttribute;
import static com.xebialabs.deployit.plugin.api.Deprecations.deprecated;
import static java.lang.reflect.Modifier.isAbstract;


public class LocalDescriptor implements Descriptor {

    static final String PLACEHOLDERS_FIELD = "placeholders";

    private Type type;
    private Class<? extends ConfigurationItem> clazz;
    private String description;
    private Metadata.ConfigurationItemRoot root;
    private List<Type> superclasses = Lists.newArrayList();
    private Set<Type> interfaces = Sets.newHashSet();
    private boolean virtual = false;

    private Map<String, PropertyDescriptor> properties = Maps.newLinkedHashMap();
    private Map<String, MethodDescriptor> controlTasks = Maps.newHashMap();
    private List<Validator<ConfigurationItem>> validators = Lists.newArrayList();
    private transient List<TypeVerification> verifications = Lists.newArrayList();

    private Type deployableType;
    private Type containerType;

    private Type generatedDeployableType;
    private Type generatedDeployableBase;
    private boolean generatedDeployableInheritsDefaultValues;
    private boolean generatedDeployableIsVirtual;
    private String generatedDeployableDescription;
    private Field syntheticPropertiesField;
    private boolean isInspectable;

    private boolean isInterface;

    private LocalDescriptor(Class<? extends ConfigurationItem> clazz) {
        this.type = Type.valueOf(clazz);
        this.virtual = clazz.isInterface();
        this.isInterface = clazz.isInterface();
        this.clazz = clazz;
    }

    private LocalDescriptor(Type type) {
        this.type = type;
    }

    static LocalDescriptor from(Class<? extends ConfigurationItem> clazz) {
        try {
            LocalDescriptor descriptor = new LocalDescriptor(clazz);
            descriptor.initMetadata();
            descriptor.scanClass();
            descriptor.initHierarchy();
            return descriptor;
        } catch (RuntimeException e) {
            throw new DescriptorException("Could not create descriptor for: " + clazz.getName(), e);
        } catch (Throwable t) {
            throw new DescriptorError("Could not create descriptor for: " + clazz.getName(), t);
        }
    }

    private void initMetadata() {
        if (virtual) {
            return;
        }

        boolean directlyAnnotated = false;
        Annotation[] declaredAnnotations = clazz.getDeclaredAnnotations();
        for (Annotation declaredAnnotation : declaredAnnotations) {
            if (declaredAnnotation.annotationType().equals(Metadata.class)) {
                directlyAnnotated = true;
            }
        }

        Metadata annotation = Preconditions.checkNotNull(clazz.getAnnotation(Metadata.class), "Class " + clazz.getName() + " or one of its ancestors does not have a @Metadata annotation");
        description = annotation.description();
        root = annotation.root();
        isInspectable = annotation.inspectable();
        virtual = (directlyAnnotated && annotation.virtual()) || Modifier.isAbstract(clazz.getModifiers());
    }

    private void scanClass() {
        findProperties();
        findControlTasks();
        findInterfaces();
        findValidations();
        findVerifications();

        Class<?> superclass = clazz.getSuperclass();
        if (superclass != null && ConfigurationItem.class.isAssignableFrom(superclass)) {
            Type supertype = Type.valueOf(superclass);
            addSuperClass(supertype);
        }

        initDeployableAndContainerTypes();
    }

    private void findVerifications() {
        for (Annotation annotation : clazz.getAnnotations()) {
            if (VerificationConverter.isVerification(annotation)) {
                this.verifications.add(VerificationConverter.<TypeVerification>makeVerification(annotation));
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void findValidations() {
        for (Annotation annotation : clazz.getAnnotations()) {
            if (ValidationRuleConverter.isRule(annotation)) {
                Validator<ConfigurationItem> v = (Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(annotation, type);
                this.validators.add(v);
            }
        }
    }

    private void findControlTasks() {
        for (Method method : clazz.getDeclaredMethods()) {
            if (method.isAnnotationPresent(ControlTask.class)) {
                MethodDescriptor from = LocalMethodDescriptor.from(this, method);
                addControlTask(from);
            }
        }
    }

    private void findProperties() {
        for (Field field : clazz.getDeclaredFields()) {
            if (field.isAnnotationPresent(Property.class)) {
                PropertyDescriptor propertyDescriptor = LocalPropertyDescriptor.from(this, field);
                addProperty(propertyDescriptor);
            }
        }
    }

    private void findInterfaces() {
        Class<?>[] clazzInterfaces = clazz.getInterfaces();
        List<Class<?>> allInterfacesFound = Lists.newArrayList();
        findAllSuperInterfaces(clazzInterfaces, allInterfacesFound);
        for (Class<?> clazzInterface : allInterfacesFound) {
            if (clazzInterface.getPackage().isAnnotationPresent(Prefix.class)) {
                addInterface(Type.valueOf(clazzInterface));
            }
        }
    }

    private void findAllSuperInterfaces(Class<?>[] childInterfaces, List<Class<?>> allInterfacesFound) {
        for (Class<?> childInterface : childInterfaces) {
            allInterfacesFound.add(childInterface);
            findAllSuperInterfaces(childInterface.getInterfaces(), allInterfacesFound);
        }
    }

    private void initDeployableAndContainerTypes() {

        if (Deployed.class.isAssignableFrom(clazz)) {
            initDeployableAndContainerTypes(Deployed.class, Deployable.class, Container.class);
        } else if (EmbeddedDeployed.class.isAssignableFrom(clazz)) {
            initDeployableAndContainerTypes(EmbeddedDeployed.class, EmbeddedDeployable.class, EmbeddedDeployedContainer.class);
        } else {
            deployableType = null;
            containerType = null;
        }
    }

    private void initDeployableAndContainerTypes(Class<? extends ConfigurationItem> deployedClass, Class<? extends ConfigurationItem> deployableClass, Class<? extends ConfigurationItem> containerClass) {
        @SuppressWarnings("unchecked")
        List<Class<?>> typeArguments = getActualTypeArguments(clazz.asSubclass(deployedClass), (Class<ConfigurationItem>)deployedClass);
        Preconditions.checkArgument(typeArguments.size() == 2, "Expected a %s and a %s, but got %s", deployableClass.getSimpleName(), containerClass.getSimpleName(), typeArguments);

        Class<?> genericDeployableClass = typeArguments.get(0);
        if (genericDeployableClass != null) {
            Preconditions.checkArgument(deployableClass.isAssignableFrom(genericDeployableClass), "Expected first item to be a %s", deployableClass.getSimpleName());
            deployableType = Type.valueOf(genericDeployableClass);
        } else {
            deployableType = null;
        }

        Class<?> genericContainerClass = typeArguments.get(1);
        if (genericContainerClass != null) {
            Preconditions.checkArgument(containerClass.isAssignableFrom(genericContainerClass), "Expected second item to be a %s", containerClass.getSimpleName());
            containerType = Type.valueOf(genericContainerClass);
        } else {
            containerType = null;
        }
    }

    static LocalDescriptor fromSynthetic(Element typeElement) {
        Type type = getRequiredTypeAttribute(typeElement, "type");

        LocalDescriptor descriptor = new LocalDescriptor(type);
        descriptor.initSynthetic(typeElement);
        descriptor.initHierarchy();

        return descriptor;
    }

    private void initSynthetic(Element typeElement) {
        description = getOptionalStringAttribute(typeElement, "description", "Description unavailable");
        virtual = getOptionalBooleanAttribute(typeElement, "virtual", false);
        isInspectable = getOptionalBooleanAttribute(typeElement, "inspectable", false);

        Type superType = getRequiredTypeAttribute(typeElement, "extends");
        addSuperClass(superType);

        parseSyntheticDeployableAndContainerType(typeElement);
        parseProperties(typeElement);
        parseControlTasks(typeElement);
        parseValidators(typeElement);
        parseVerifications(typeElement);
    }

    private void parseVerifications(final Element typeElement) {
        forEach(childByName(typeElement, equalTo(VerificationConverter.VERIFICATION_ELEMENT_NAME)), new Closure<Element>() {
            @Override
            public void call(Element element) {
                verifications.add(VerificationConverter.<TypeVerification>makeVerification(element, type));
            }
        });
    }

    private void parseValidators(Element typeElement) {
        forEach(childByName(typeElement, equalTo(ValidationRuleConverter.RULE_ELEMENT_NAME)), new Closure<Element>() {
            @SuppressWarnings("unchecked")
            @Override
            public void call(Element element) {
                validators.add((Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(element, type));
            }
        });
    }

    private void parseSyntheticDeployableAndContainerType(Element typeElement) {
        deployableType = getOptionalTypeAttribute(typeElement, "deployable-type");
        containerType = getOptionalTypeAttribute(typeElement, "container-type");
        Iterator<Element> generateElements = childByName(typeElement, equalTo("generate-deployable"));
        if (generateElements.hasNext()) {
            Element generateDeployable = generateElements.next();
            generatedDeployableType = getRequiredTypeAttribute(generateDeployable, "type");
            generatedDeployableBase = getRequiredTypeAttribute(generateDeployable, "extends");
            generatedDeployableInheritsDefaultValues = getOptionalBooleanAttribute(generateDeployable, "copy-default-values", false);
            generatedDeployableIsVirtual = getOptionalBooleanAttribute(generateDeployable, "virtual", false);
            generatedDeployableDescription = getOptionalStringAttribute(generateDeployable, "description", "Description unavailable");
        }
    }

    void parseTypeModification(Element element) {
        parseProperties(element);
        parseControlTasks(element);
        parseValidators(element);
        parseVerifications(element);
    }

    private void parseControlTasks(final Element element) {
        forEach(childByName(element, equalTo("method")), new Closure<Element>() {
            @Override
            public void call(Element element) {
                MethodDescriptor controlTask = LocalMethodDescriptor.from(LocalDescriptor.this, element);
                verifyNewControlTask(controlTask);
                addControlTask(controlTask);
            }
        });
    }

    private void parseProperties(final Element element) {
        forEach(childByName(element, equalTo("property")), new Closure<Element>() {
            @Override
            public void call(Element element) {
                String name = getRequiredStringAttribute(element, "name");
                LocalPropertyDescriptor newDesc = LocalPropertyDescriptor.from(LocalDescriptor.this, element);
                PropertyDescriptor oldDesc = properties.get(name);
                if (oldDesc != null) {
                    newDesc.overrideWith((LocalPropertyDescriptor) oldDesc);
                }
                addProperty(newDesc);
            }
        });
    }

    private void verifyNewControlTask(MethodDescriptor controlTask) {
        if (controlTasks.containsKey(controlTask.getName())) {
            throw new IllegalStateException("Cannot override existing Control Task [" + controlTask.getFqn() + "] with a synthetic one.");
        }
    }

    static LocalDescriptor fromGenerateDeployable(LocalDescriptor deployedDescriptor) {
        LocalDescriptor deployableDescriptor = new LocalDescriptor(deployedDescriptor.generatedDeployableType);
        deployableDescriptor.addSuperClass(deployedDescriptor.generatedDeployableBase);
        deployableDescriptor.initDeployableFromDeployed(deployedDescriptor);
        deployableDescriptor.initHierarchy();
        return deployableDescriptor;
    }

    private void initDeployableFromDeployed(LocalDescriptor deployedDescriptor) {
        if (deployedDescriptor.generatedDeployableDescription == null || deployedDescriptor.generatedDeployableDescription.equals("Description unavailable")) {
            description = deployedDescriptor.getDescription() + " (deployable)";
        } else {
            description = deployedDescriptor.generatedDeployableDescription;
        }

        virtual = deployedDescriptor.generatedDeployableIsVirtual;

        for (PropertyDescriptor deployedPropertyDescriptor : deployedDescriptor.getPropertyDescriptors()) {
            String name = deployedPropertyDescriptor.getName();
            boolean isUdmField = name.equals(Deployed.DEPLOYABLE_FIELD) || name.equals(Deployed.CONTAINER_FIELD)
                    || name.equals(PLACEHOLDERS_FIELD);
            final PropertyKind kind = deployedPropertyDescriptor.getKind();
            boolean isReferenceField = kind == PropertyKind.CI || kind == PropertyKind.SET_OF_CI || kind == PropertyKind.LIST_OF_CI;
            if (isUdmField || isReferenceField || deployedPropertyDescriptor.isHidden()) {
                continue;
            }

            PropertyDescriptor generatedDeployablePropertyDescriptor = LocalPropertyDescriptor.generateDeployableFrom(this, deployedPropertyDescriptor);
            if (deployedDescriptor.generatedDeployableInheritsDefaultValues) {
                GlobalContext.register(generatedDeployablePropertyDescriptor, GlobalContext.lookup(deployedPropertyDescriptor));
            }
            addProperty(generatedDeployablePropertyDescriptor);
        }
    }

    public static LocalDescriptor fromGeneratedParameters(final Type type, final Element methodDef) {
        LocalDescriptor localDescriptor = new LocalDescriptor(type);
        localDescriptor.initParameters(methodDef);
        localDescriptor.initHierarchy();
        return localDescriptor;
    }

    private void initParameters(final Element methodDef) {
        addSuperClass(Type.valueOf(Parameters.class));
        String name = getRequiredStringAttribute(methodDef, "name");
        description = "Generated type for method " + name + ".";
        root = Metadata.ConfigurationItemRoot.NESTED;
        Element parameters = childByName(methodDef, equalTo("parameters")).next();
        for (Element parameter : childrenByName(parameters, equalTo("parameter"))) {
            LocalPropertyDescriptor propDesc = LocalPropertyDescriptor.from(LocalDescriptor.this, parameter);
            addProperty(propDesc);
        }
    }


    void initHierarchy() {
        if (!superclasses.isEmpty()) {
            Type toInitFrom = superclasses.get(0);

            LocalDescriptor superDesc = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(toInitFrom);
            if (superDesc == null) {
                throw new IllegalStateException("Cannot build type hierarchy for " + getType() + " because one of its supertypes cannot be found: " + toInitFrom + " not found");
            }

            // For synthetic types, root and clazz might not be set yet.
            if (root == null) {
                root = superDesc.getRoot();
            }

            if (clazz == null) {
                clazz = superDesc.clazz;
            }

            inheritPropertyDescriptors(properties, superDesc.properties);
            inheritControlTasks(controlTasks, superDesc.controlTasks);
            inheritValidators(validators, superDesc.validators);
            inheritVerifications(verifications, superDesc.verifications);

            for (Type superclass : superDesc.superclasses) {
                addSuperClass(superclass);
            }
            for (Type intf : superDesc.interfaces) {
                addInterface(intf);
            }

            if (deployableType == null) {
                deployableType = superDesc.getDeployableType();
            }

            if (containerType == null) {
                containerType = superDesc.getContainerType();
            }
        }

        if (!clazz.isInterface() && !Type.valueOf("api.ValidatedConfigurationItem").equals(type)) {
            syntheticPropertiesField = searchField(clazz, ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD);
            syntheticPropertiesField.setAccessible(true);
        }
    }

    void verify(final Verifications verifications) {
        verifySyntheticPropertiesField(verifications);
        verifyReferenceTypes(verifications);
        verifyNonArtifactDoesNotHavePlaceholders(verifications);
        verifyArtifactInterfaces(verifications);
        if (!isVirtual()) {
            verifyNoArgsConstructor(verifications);
            verifyJavaClassIsNotAbstract(verifications);
            verifyDeployedHasDeployableAndContainerType(verifications);
            verifyDeployedHasCorrectInheritance(verifications);
            verifyHiddenRequiredPropertiesHaveDefaultValue(verifications);
            verifyControlTasks(verifications);
        }
        if (!isInspectable()) {
            verifyNoInspectionProperties(verifications);
        }
        verifyExtendsBaseConfigurationItem(verifications);

        for (TypeVerification verification : this.verifications) {
            verification.verify(this, new VerificationContext() {
                @Override
                public void error(final String message, final Object... params) {
                    verifications.verify(type, false, message, params);
                }
            });
        }

        for (PropertyDescriptor propertyDescriptor : properties.values()) {
            ((LocalPropertyDescriptor) propertyDescriptor).verify(verifications);
        }

        verifyAsContainmentProperties(verifications);
    }

    private void verifyNoArgsConstructor(Verifications verifications) {
        Constructor<?>[] declaredConstructors = clazz.getDeclaredConstructors();
        for (Constructor<?> declaredConstructor : declaredConstructors) {
            if (declaredConstructor.getParameterTypes().length == 0) {
                return;
            }
        }
        verifications.verify(type, false, "ConfigurationItem should have a no-arguments constructor.");
    }

    private void verifyAsContainmentProperties(Verifications verifications) {
        FluentIterable<PropertyDescriptor> from = FluentIterable.from(properties.values());
        FluentIterable<PropertyDescriptor> filteredCiProperties = from.filter(new Predicate<PropertyDescriptor>() {
            public boolean apply(PropertyDescriptor input) {
                return input.isAsContainment() && input.getKind() == PropertyKind.CI;
            }
        });
        verifications.verify(filteredCiProperties.size() <= 1, "Can only have at most 1 asContainment property with kind = 'ci', got %s", filteredCiProperties);
    }

    private void verifyExtendsBaseConfigurationItem(@SuppressWarnings("UnusedParameters") final Verifications verifications) {
        if (clazz != null) {
            if (!BaseConfigurationItem.class.isAssignableFrom(clazz)) {
                deprecated("ConfigurationItem [%s] does not extend BaseConfigurationItem, will be required in Deployit 4.0", type);
                logger.warn("Concurrent modifications cannot be detected on Configuration Items of type [{}] because is does not extend BaseConfigurationItem.", type);
            }
        }
    }

    private void verifyNoInspectionProperties(Verifications verifications) {
        List<PropertyDescriptor> inspectableProperties = newArrayList(filter(properties.values(), new Predicate<PropertyDescriptor>() {
            @Override
            public boolean apply(PropertyDescriptor input) {
                return input.isInspectionProperty();
            }
        }));
        verifications.verify(type, inspectableProperties.isEmpty(), "Type [%s] is not inspectable but has @InspectionProperty properties %s", type, inspectableProperties);
    }

    private void verifySyntheticPropertiesField(Verifications verifications) {
        verifications.verify(type, syntheticPropertiesField != null, "Synthetic properties field should be set");
        if (syntheticPropertiesField != null) {
            verifications.verify(type, syntheticPropertiesField.getType().isAssignableFrom(Map.class), "Synthetic properties field should be Map<String, Object>, not: %s", syntheticPropertiesField.getType());
        }
    }

    private void verifyControlTasks(Verifications verifications) {
        for (MethodDescriptor controlTask : controlTasks.values()) {
            ((LocalMethodDescriptor) controlTask).verify(verifications);
        }
    }


    private void verifyReferenceTypes(Verifications verifications) {
        for (PropertyDescriptor p : properties.values()) {
            final PropertyKind kind = p.getKind();
            if (kind == PropertyKind.CI || kind == PropertyKind.SET_OF_CI || kind == PropertyKind.LIST_OF_CI) {
                verifications.verify(type, isValidReferencedType(p.getReferencedType()), "Property %s of type %s refers to non-existing type %s", p.getName(), ((LocalPropertyDescriptor) p).getDeclaringDescriptor().getType(), p.getReferencedType());
            }
        }
    }

    private boolean isValidReferencedType(Type referencedType) {
        if (LocalDescriptorRegistry.exists(referencedType))
            return true;

        for (Descriptor d : LocalDescriptorRegistry.getDescriptors()) {
            if (d.getSuperClasses().contains(referencedType)) {
                return true;
            }
            if (d.getInterfaces().contains(referencedType)) {
                return true;
            }
        }

        return false;
    }

    private void verifyNonArtifactDoesNotHavePlaceholders(Verifications verifications) {
        if (!isAssignableTo(Artifact.class)) {
            for (PropertyDescriptor propertyDescriptor : properties.values()) {
                verifications.verify(type, !propertyDescriptor.getName().equals(PLACEHOLDERS_FIELD), "Cannot have a field 'placeholders' as 'udm.Artifact' is not implemented.");
            }
        }
    }

    private void verifyArtifactInterfaces(Verifications verifications) {
        if (isAssignableTo(Deployable.class) && isAssignableTo(Artifact.class)) {
            verifications.verify(type, isAssignableTo(SourceArtifact.class), "Implements both 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.SourceArtifact' interface");
            verifications.verify(type, isAssignableTo(DeployableArtifact.class), "Implements the 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.DeployableArtifact' interface");
        }
        if (isAssignableTo(Deployed.class) && isAssignableTo(Artifact.class)) {
            verifications.verify(type, isAssignableTo(DerivedArtifact.class), "Implements the 'udm.Deployed' and 'udm.Artifact' interface. Must also implement the 'udm.DerivedArtifact' interface");
        }
    }

    private void verifyJavaClassIsNotAbstract(Verifications verifications) {
        verifications.verify(type, !isAbstract(clazz.getModifiers()), "Non-virtual type %s has an abstract Java class %s", type, clazz.getName());
    }

    private void verifyDeployedHasDeployableAndContainerType(Verifications verifications) {
        Type deployedType = (isAssignableTo(Deployed.class) ? Type.valueOf(Deployed.class) : (isAssignableTo(EmbeddedDeployed.class) ? Type.valueOf(EmbeddedDeployed.class) : null));
        if (deployedType != null) {
            verifications.verify(type, getDeployableType() != null, "Non-virtual type %s is a sub-type of %s but does not have a deployable-type", getType(), deployedType);
            verifications.verify(type, getContainerType() != null, "Non-virtual type %s is a sub-type of %s but does not have a container-type", getType(), deployedType);
        } else {
            verifications.verify(type, getDeployableType() == null, "Non-virtual type %s should not have a deployable-type", getType());
            verifications.verify(type, getContainerType() == null, "Non-virtual type %s should not have a container-type", getType());
        }
    }

    private void verifyDeployedHasCorrectInheritance(Verifications verifications) {
        Type deployedType = Type.valueOf(Deployed.class);
        if (isAssignableTo(deployedType)) {
            Type directSuperType = superclasses.get(0);
            Descriptor superDescriptor = directSuperType.getDescriptor();
            Type superDeployable = superDescriptor.getDeployableType();
            Type superContainer = superDescriptor.getContainerType();
            verifications.verify(type, superDeployable == null || deployableType.instanceOf(superDeployable), "Type %s inherits from %s<%s, %s> but doesn't have a deployable subtype (%s).", type, directSuperType, superDeployable, superContainer, deployableType);
            verifications.verify(type, superContainer == null || containerType.instanceOf(superContainer), "Type %s inherits from %s<%s, %s> but doesn't have a container subtype (%s).", type, directSuperType, superDeployable, superContainer, containerType);
        }
    }

    private void verifyHiddenRequiredPropertiesHaveDefaultValue(Verifications verifications) {
        for (PropertyDescriptor p : properties.values()) {
            if (p.isHidden() && p.isRequired()) {
                verifications.verify(type, p.getDefaultValue() != null, "Hidden required property %s of non-virtual type %s must have a default value", p.getName(), getType());
            }
        }
    }

    private void inheritValidators(List<Validator<ConfigurationItem>> dest, List<Validator<ConfigurationItem>> source) {
        dest.addAll(source);
    }

    private void inheritVerifications(List<TypeVerification> dest, List<TypeVerification> src) {
        dest.addAll(src);
    }

    private void inheritControlTasks(Map<String, MethodDescriptor> dest, Map<String, MethodDescriptor> source) {
        for (Map.Entry<String, MethodDescriptor> sourceEntry : source.entrySet()) {
            if (!dest.containsKey(sourceEntry.getKey())) {
                dest.put(sourceEntry.getKey(), new LocalMethodDescriptor((LocalMethodDescriptor) sourceEntry.getValue(), this));
            } else {
                logger.warn("Not inheriting ControlTask [{}] on [{}]", new Object[]{sourceEntry.getValue().getFqn(), type});
            }
        }
    }

    private void inheritPropertyDescriptors(Map<String, PropertyDescriptor> dest, Map<String, PropertyDescriptor> source) {
        Map<String, PropertyDescriptor> myDescriptors = Maps.newLinkedHashMap(dest);
        dest.clear();
        for (Map.Entry<String, PropertyDescriptor> sourceEntry : source.entrySet()) {
            if (!myDescriptors.containsKey(sourceEntry.getKey())) {
                dest.put(sourceEntry.getKey(), new LocalPropertyDescriptor((LocalPropertyDescriptor) sourceEntry.getValue(), this));
            } else {
                ((LocalPropertyDescriptor) myDescriptors.get(sourceEntry.getKey())).overrideWith((LocalPropertyDescriptor) sourceEntry.getValue());
            }
        }
        dest.putAll(myDescriptors);
    }

    private void addSuperClass(Type supertype) {
        superclasses.add(supertype);
    }

    private void addInterface(Type intf) {
        interfaces.add(intf);
    }

    private void addProperty(PropertyDescriptor propertyDescriptor) {
        properties.put(propertyDescriptor.getName(), propertyDescriptor);
    }

    private void addControlTask(MethodDescriptor from) {
        controlTasks.put(from.getName(), from);
    }

    @Override
    public Type getType() {
        return type;
    }

    @Override
    public Class<?> getClazz() {
        return clazz;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Metadata.ConfigurationItemRoot getRoot() {
        return root;
    }

    @Override
    public Collection<PropertyDescriptor> getPropertyDescriptors() {
        return newArrayList(properties.values());
    }

    @Override
    public PropertyDescriptor getPropertyDescriptor(String name) {
        return properties.get(name);
    }

    @Override
    public MethodDescriptor getControlTask(String name) {
        return controlTasks.get(name);
    }

    @Override
    public Collection<MethodDescriptor> getControlTasks() {
        return controlTasks.values();
    }

    @Override
    public boolean isAssignableTo(Class<?> clazz) {
        return isAssignableTo(Type.valueOf(clazz));
    }

    @Override
    public boolean isAssignableTo(Type type) {
        return this.type.isSubTypeOf(type) || this.type.equals(type);
    }

    @Override
    public List<Type> getSuperClasses() {
        return superclasses;
    }

    @Override
    public Set<Type> getInterfaces() {
        return interfaces;
    }

    @Override
    public boolean isVirtual() {
        return virtual;
    }

    public boolean isInterface() {
        return isInterface;
    }

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<String>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        if (item == null) {
            return other == null;
        }

        if (!getType().equals(item.getType()) || !getType().equals(other.getType())) {
            return false;
        }

        if (!itemsBeingCompared.add(item.getId())) {
            return true;
        }

        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            if (!((LocalPropertyDescriptor) pd).areEqual(item, other, itemsBeingCompared)) {
                return false;
            }
        }
        return true;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> T newInstance() {
        if (virtual) {
            throw new IllegalArgumentException("Cannot instantiate class for " + type + " because it is virtual");
        }

        try {
            Field typeField = searchField(clazz, ConfigurationItem.TYPE_FIELD);
            typeField.setAccessible(true);
            T t = (T) clazz.newInstance();
            typeField.set(t, type);

            prefillDefaultProperties(t);

            return t;
        } catch (InstantiationException exc) {
            throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
        } catch (IllegalAccessException exc) {
            throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
        }
    }

    @Override
    public <T extends ConfigurationItem> T newInstance(String id) {
        T configurationItem = newInstance();
        configurationItem.setId(id);
        return configurationItem;
    }

    private <T extends ConfigurationItem> void prefillDefaultProperties(T t) {
        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            LocalPropertyDescriptor lpd = (LocalPropertyDescriptor) pd;
            if (pd.getDefaultValue() != null) {
                pd.set(t, lpd.getDefaultValue());
            } else {
                pd.set(t, lpd.emptyValue());
            }
        }
    }

    @Override
    public String toString() {
        return "Descriptor[" + type + "]";
    }

    @Override
    public Type getDeployableType() {
        return deployableType;
    }

    @Override
    public Type getContainerType() {
        return containerType;
    }

    Field getSyntheticPropertiesField() {
        return syntheticPropertiesField;
    }

    @Override
    public List<ValidationMessage> validate(final ConfigurationItem ci) {
        final ArrayList<ValidationMessage> messages = Lists.newArrayList();
        for (PropertyDescriptor propertyDescriptor : properties.values()) {
            ((LocalPropertyDescriptor) propertyDescriptor).validate(ci, messages);
        }
        for (Validator<ConfigurationItem> validator : validators) {
            validator.validate(ci, new ValidationContext() {
                @Override
                public void error(String message, Object... params) {
                    messages.add(new ValidationMessage(ci.getId(), null, String.format(message, params)));
                }
            });
        }
        return messages;
    }

    @Override
    public boolean isInspectable() {
        return isInspectable;
    }

    @SuppressWarnings("serial")
    private static class DescriptorException extends RuntimeException {
        public DescriptorException(String s, RuntimeException e) {
            super(s, e);
        }
    }

    @SuppressWarnings("serial")
    private static class DescriptorError extends Error {
        public DescriptorError(String s, Throwable e) {
            super(s, e);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalDescriptor.class);
}
