package com.xebialabs.deployit.booter.local;

import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Class responsible for booting all the plugins. Responsible for:
 * - Registering all the Descriptors
 * - Setting up the Global Context.
 */
public class LocalBooter {

    private static final AtomicBoolean isBooted = new AtomicBoolean(false);
    private static final AtomicBoolean initialized = new AtomicBoolean(false);

    /**
     * Boot the Deployit Plugin System. Setting everything up.
     */
    public static void boot() {
        try {
            boot(true);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the Deployit Plugin System, but without initializing the global context with user overridden default values.
     * Useful for tests where you want to have control which default values a CI has.
     */
    public static void bootWithoutGlobalContext() {
        boot(false);
    }

    private synchronized static void boot(boolean withGlobalContext) {
        if (!isBooted.getAndSet(true)) {
            Scannit.boot(Configuration.config()
                    .scan("com.xebialabs")
                    .scan("ext.deployit") // Deployit Extensions
                    .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner()));
            DelegateRegistry.boot();
            TypeInitializer.init();
            SyntheticInitializer.init();
            DescriptorHierarchyInitializer.init();
            DeployableGenerator.init();
            Verifications verifications = new Verifications();
            DelegateRegistry.verify(verifications);
            TypeInitializer.verifyTypes(verifications);
            verifications.done();
            if (withGlobalContext) {
                GlobalContextInitializer.init();
            }
            PluginVersions.init();
            initialized.set(true);
        }

        if (!initialized.get()) {
            throw new IllegalStateException("The DescriptorRegistry has been booted, but is not initialized. Please check the logs for any errors.");
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalBooter.class);
}
