package com.xebialabs.deployit.booter.local.generator;

import com.xebialabs.deployit.booter.local.TypeDefinitions;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

public class TypeGenerators {

    private static final Logger logger = LoggerFactory.getLogger(TypeGenerators.class);

    private final Map<String, TypeGenerator> generatorsByName = new HashMap<>();

    public TypeGenerators(Scannit scannit) {
        scanTypeGenerators(scannit);
    }

    private void scanTypeGenerators(Scannit scannit) {
        scannit.getSubTypesOf(TypeGenerator.class).forEach(g -> {
            TypeGenerator typeGenerator = null;
            try {
                typeGenerator = g.newInstance();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            generatorsByName.put(typeGenerator.getName(), typeGenerator);
        });
    }

    public void generateTypeDefinitions(TypeDefinitions typeDefinitions) {
        generatorsByName.values().forEach(tg -> executeWithDependencies(typeDefinitions, tg, new ArrayList<>(), new ArrayList<>()));
    }

    private void executeWithDependencies(TypeDefinitions typeDefinitions, TypeGenerator typeGenerator, List<String> alreadyExecuted, List<String> wanted) {
        if (alreadyExecuted.contains(typeGenerator.getName())) {
            return;
        }
        if (wanted.contains(typeGenerator.getName())) {
            throw new RuntimeException("There is a circular dependency in TypeGenerators " + Arrays.toString(wanted.toArray(new String[wanted.size()])));
        }
        wanted.add(typeGenerator.getName());
        if (alreadyExecuted.containsAll(typeGenerator.dependsOn())) {
            execute(typeDefinitions, typeGenerator, alreadyExecuted, wanted);
        } else {
            typeGenerator.dependsOn().forEach(dep -> executeWithDependencies(typeDefinitions, typeGeneratorByName(dep), alreadyExecuted, wanted));
        }
    }

    private void execute(TypeDefinitions typeDefinitions, TypeGenerator typeGenerator, List<String> alreadyExecuted, List<String> wanted) {
        logger.info("Executing " + typeGenerator.getName());
        alreadyExecuted.add(typeGenerator.getName());
        wanted.remove(typeGenerator.getName());
        typeGenerator.generateAndRegister(typeDefinitions);
    }

    private TypeGenerator typeGeneratorByName(String name) {
        return Optional.ofNullable(generatorsByName.get(name)).orElseThrow(() -> new RuntimeException("Type generator " + name + " is not found"));
    }
}
