package com.xebialabs.deployit.booter.local;

import java.lang.reflect.Constructor;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

import static java.lang.reflect.Modifier.isAbstract;

final class DescriptorVerification {

    static final String PLACEHOLDERS_FIELD = "placeholders";

    static void verify(final Verifications verifications, Descriptor descriptor) {
        verifyReferenceTypes(verifications, descriptor);
        verifyNonArtifactDoesNotHavePlaceholders(verifications, descriptor);
        verifyArtifactInterfaces(verifications, descriptor);
        if (!descriptor.isVirtual()) {
            verifyNoArgsConstructor(verifications, descriptor);
            verifyJavaClassIsNotAbstract(verifications, descriptor);
            verifyDeployedHasDeployableAndContainerType(verifications, descriptor);
            verifyDeployedHasCorrectInheritance(verifications, descriptor);
            verifyHiddenRequiredPropertiesHaveDefaultValue(verifications, descriptor);
            verifyControlTasks(verifications, descriptor);
        }
        if (!descriptor.isInspectable()) {
            verifyNoInspectionProperties(verifications, descriptor);
        }
        verifyExtendsBaseConfigurationItem(verifications, descriptor);

        for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
            ((LocalPropertyDescriptor) propertyDescriptor).verify(verifications);
        }

        verifyAsContainmentProperties(verifications, descriptor);
    }

    static private void verifyNoArgsConstructor(Verifications verifications, Descriptor descriptor) {
        Constructor<?>[] declaredConstructors = descriptor.getClazz().getDeclaredConstructors();
        for (Constructor<?> declaredConstructor : declaredConstructors) {
            if (declaredConstructor.getParameterTypes().length == 0) {
                return;
            }
        }
        verifications.verify(descriptor.getType(), false, "ConfigurationItem should have a no-arguments constructor.");
    }

    static private void verifyAsContainmentProperties(Verifications verifications, Descriptor descriptor) {
        Stream<PropertyDescriptor> from = descriptor.getPropertyDescriptors().stream();
        List<PropertyDescriptor> filteredCiProperties = from.filter(input -> input.isAsContainment() && input.getKind() == PropertyKind.CI).collect(Collectors.toList());
        verifications.verify(filteredCiProperties.size() <= 1, "Can only have at most 1 asContainment property with kind = 'ci', got %s", filteredCiProperties);
    }

    static private void verifyExtendsBaseConfigurationItem(@SuppressWarnings("UnusedParameters") final Verifications verifications, Descriptor descriptor) {
        if (descriptor.getClazz() != null) {
            verifications.verify(BaseConfigurationItem.class.isAssignableFrom(descriptor.getClazz()), "ConfigurationItem [%s] does not extend BaseConfigurationItem", descriptor.getType());
        }
    }

    static private void verifyNoInspectionProperties(Verifications verifications, Descriptor descriptor) {
        Type type = descriptor.getType();
        List<PropertyDescriptor> inspectableProperties = descriptor.getPropertyDescriptors().stream().filter(PropertyDescriptor::isInspectionProperty).collect(Collectors.toList());
        verifications.verify(type, inspectableProperties.isEmpty(), "Type [%s] is not inspectable but has @InspectionProperty properties %s", type, inspectableProperties);
    }

    static private void verifyControlTasks(Verifications verifications, Descriptor descriptor) {
        for (MethodDescriptor controlTask : descriptor.getControlTasks()) {
            ((LocalMethodDescriptor) controlTask).verify(verifications);
        }
    }


    static private void verifyReferenceTypes(Verifications verifications, Descriptor descriptor) {
        for (PropertyDescriptor p : descriptor.getPropertyDescriptors()) {
            final PropertyKind kind = p.getKind();
            if (kind == PropertyKind.CI || kind == PropertyKind.SET_OF_CI || kind == PropertyKind.LIST_OF_CI) {
                verifications.verify(descriptor.getType(), isValidReferencedType(p.getReferencedType()), "Property %s of type %s refers to non-existing type %s", p.getName(), ((LocalPropertyDescriptor) p).getDeclaringDescriptor().getType(), p.getReferencedType());
            }
        }
    }

    static private boolean isValidReferencedType(Type referencedType) {
        if (referencedType.exists())
            return true;
        final DescriptorRegistryId descriptorRegistryId = referencedType.getTypeSource();
        final IDescriptorRegistry descriptorRegistry = DescriptorRegistry.getDescriptorRegistry(descriptorRegistryId);
        for (Descriptor d : descriptorRegistry.getDescriptors()) {
            if (d.getSuperClasses().contains(referencedType)) {
                return true;
            }
            if (d.getInterfaces().contains(referencedType)) {
                return true;
            }
        }

        return false;
    }

    static private void verifyNonArtifactDoesNotHavePlaceholders(Verifications verifications, Descriptor descriptor) {
        if (!descriptor.isAssignableTo(Artifact.class)) {
            for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
                verifications.verify(descriptor.getType(), !propertyDescriptor.getName().equals(PLACEHOLDERS_FIELD), "Cannot have a field 'placeholders' as 'udm.Artifact' is not implemented.");
            }
        }
    }

    static private void verifyArtifactInterfaces(Verifications verifications, Descriptor descriptor) {
        Type type = descriptor.getType();
        if (descriptor.isAssignableTo(Deployable.class) && descriptor.isAssignableTo(Artifact.class)) {
            verifications.verify(type, descriptor.isAssignableTo(SourceArtifact.class), "Implements both 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.SourceArtifact' interface");
            verifications.verify(type, descriptor.isAssignableTo(DeployableArtifact.class), "Implements the 'udm.Deployable' and 'udm.Artifact' interface. Must also implement the 'udm.DeployableArtifact' interface");
        }
        if (descriptor.isAssignableTo(Deployed.class) && descriptor.isAssignableTo(Artifact.class)) {
            verifications.verify(type, descriptor.isAssignableTo(DerivedArtifact.class), "Implements the 'udm.Deployed' and 'udm.Artifact' interface. Must also implement the 'udm.DerivedArtifact' interface");
        }
    }

    static private void verifyJavaClassIsNotAbstract(Verifications verifications, Descriptor descriptor) {
        verifications.verify(descriptor.getType(), !isAbstract(descriptor.getClazz().getModifiers()), "Non-virtual type %s has an abstract Java class %s", descriptor.getType(), descriptor.getClazz().getName());
    }

    static private void verifyDeployedHasDeployableAndContainerType(Verifications verifications, Descriptor descriptor) {
        Type type = descriptor.getType();
        final DescriptorRegistryId descriptorRegistryId = type.getTypeSource();
        final IDescriptorRegistry descriptorRegistry = DescriptorRegistry.getDescriptorRegistry(descriptorRegistryId);
        final Type embeddedDeployedType = descriptor.isAssignableTo(EmbeddedDeployed.class) ? descriptorRegistry.lookupType(EmbeddedDeployed.class) : null;
        Type deployedType = descriptor.isAssignableTo(Deployed.class) ? descriptorRegistry.lookupType(Deployed.class) : embeddedDeployedType;
        Type deplyableType = descriptor.getDeployableType();
        Type containerType = descriptor.getContainerType();
        if (deployedType != null) {
            verifications.verify(type, deplyableType != null, "Non-virtual type %s is a sub-type of %s but does not have a deployable-type", type, deployedType);
            verifications.verify(type, containerType != null, "Non-virtual type %s is a sub-type of %s but does not have a container-type", type, deployedType);
        } else {
            verifications.verify(type, deplyableType == null, "Non-virtual type %s should not have a deployable-type", type);
            verifications.verify(type, containerType == null, "Non-virtual type %s should not have a container-type", type);
        }
    }

    static private void verifyDeployedHasCorrectInheritance(Verifications verifications, Descriptor descriptor) {
        final DescriptorRegistryId descriptorRegistryId = descriptor.getType().getTypeSource();
        final IDescriptorRegistry descriptorRegistry = DescriptorRegistry.getDescriptorRegistry(descriptorRegistryId);
        Type deployedType = descriptorRegistry.lookupType(Deployed.class);
        Type type = descriptor.getType();
        Type deployableType = descriptor.getDeployableType();
        Type containerType = descriptor.getContainerType();
        if (descriptor.isAssignableTo(deployedType)) {
            Type directSuperType = descriptor.getSuperClasses().get(0);
            Descriptor superDescriptor = directSuperType.getDescriptor();
            Type superDeployable = superDescriptor.getDeployableType();
            Type superContainer = superDescriptor.getContainerType();
            verifications.verify(descriptor.getType(), superDeployable == null || descriptor.getDeployableType().instanceOf(superDeployable),
                    "Type %s inherits from %s<%s, %s> but doesn't have a deployable subtype (%s).", type, directSuperType, superDeployable, superContainer, deployableType);
            verifications.verify(descriptor.getType(), superContainer == null || descriptor.getContainerType().instanceOf(superContainer),
                    "Type %s inherits from %s<%s, %s> but doesn't have a container subtype (%s).", type, directSuperType, superDeployable, superContainer, containerType);
        }
    }

    static private void verifyHiddenRequiredPropertiesHaveDefaultValue(Verifications verifications, Descriptor descriptor) {
        Type type = descriptor.getType();
        descriptor.getPropertyDescriptors().stream()
                .filter(p -> p.isHidden() && p.isRequired())
                .forEach(p -> verifications.verify(type, p.getDefaultValue() != null,
                        "Hidden required property %s of non-virtual type %s must have a default value", p.getName(), type));
    }

}
