package com.xebialabs.xlplatform.synthetic.yaml;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLMapper;
import com.networknt.schema.JsonSchema;
import com.networknt.schema.JsonSchemaFactory;
import com.networknt.schema.SpecVersion;
import com.networknt.schema.ValidationMessage;

import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;

/**
 * Loads type definitions in Yaml and exposes them in the same format as {@link com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument}
 * to be picked up by the {@code TypeSystemBootstrapper}.
 */
public class TypeDefinitionYamlDocument {

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitionYamlDocument.class);
    public static final ObjectMapper YAML_PARSER = new YAMLMapper(new YAMLFactory());

    private final List<TypeSpecification> types = new ArrayList<>();

    private TypeDefinitionYamlDocument() {
    }

    public List<TypeSpecification> getTypes() {
        return types;
    }

    private void readTypes(JsonNode node) {
        if (node.has("types")) {
            JsonNode typeNodes = node.get("types");
            Iterator<String> iter = typeNodes.fieldNames();
            while (iter.hasNext()) {
                String typeName = iter.next();
                types.add(new JsonTypeSpecification(typeName, typeNodes.get(typeName)));
            }
        }
    }

    //
    // Parse & validate
    //

    public static TypeDefinitionYamlDocument read(URL typeYaml) throws IOException {

        TypeDefinitionYamlDocument typeDefinitionYaml = new TypeDefinitionYamlDocument();

        JsonNode node = readYaml(typeYaml);
        validate(node, Optional.of(typeYaml.getFile()));

        typeDefinitionYaml.readTypes(node);

        return typeDefinitionYaml;
    }

    public static TypeDefinitionYamlDocument read(String yamlContent) throws IOException {

        TypeDefinitionYamlDocument typeDefinitionYaml = new TypeDefinitionYamlDocument();

        JsonNode node = readYaml(yamlContent);
        validate(node, Optional.empty());

        typeDefinitionYaml.readTypes(node);

        return typeDefinitionYaml;
    }

    private static JsonNode readYaml(URL yaml) throws IOException {
        return YAML_PARSER.readTree(yaml);
    }

    private static JsonNode readYaml(String yaml) throws IOException {
        return YAML_PARSER.readTree(yaml);
    }

    private static void validate(JsonNode node, Optional<String> file) throws IOException {
        Set<ValidationMessage> messages = validateYaml(node);

        messages.forEach(message ->
                logger.error(message.toString()));

        if (!messages.isEmpty()) {
            if (file.isPresent()) {
                throw new IllegalArgumentException("One or more errors were found while parsing " + file.get());
            } else {
                throw new IllegalArgumentException("One or more errors were found while parsing type-definitions");
            }
        }
    }

    public static Set<ValidationMessage> validateYaml(String file) throws IOException {
        return validateYaml(getResourceUrl(file));
    }

    public static Set<ValidationMessage> validateYaml(URL yaml) throws IOException {
        return validateYaml(readYaml(yaml));
    }

    private static Set<ValidationMessage> validateYaml(JsonNode yaml) throws IOException {

        JsonSchemaFactory factory = JsonSchemaFactory.builder(JsonSchemaFactory.getInstance(SpecVersion.VersionFlag.V202012))
                .objectMapper(YAML_PARSER).build();

        JsonSchema schema = null;
        try {
            schema = factory.getSchema(getResourceUrl("type-definition-schema.json").toURI());
        } catch (URISyntaxException e) {
            throw new IOException(e);
        }

        return schema.validate(yaml);
    }

    private static URL getResourceUrl(String file) {
        return MODULE$.classLoader().getResource(file);
    }
}
