package com.xebialabs.deployit.booter.local;

import java.io.File;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;

import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

/**
 * Class responsible for booting all the plugins. Responsible for: - Registering all the Descriptors - Setting up the
 * Global Context.
 */
public class LocalBooter {

    private static final AtomicBoolean isBooted = new AtomicBoolean(false);
    private static final AtomicBoolean initialized = new AtomicBoolean(false);

    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void boot() {
        try {
            boot(true, Collections.<String>emptyList(), false);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void boot(File propertiesFile) {
        try {
            boot(true, Collections.<String>emptyList(), false, propertiesFile, null);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    public static void boot(File propertiesFile, GlobalContextManager globalContextManager) {
        try {
            boot(true, Collections.<String>emptyList(), false, propertiesFile, globalContextManager);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System. Setting everything up. You can also specify extra packages to scan for classes
     * with XL Deploy annotations.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void boot(List<String> packagesToScan) {
        try {
            boot(true, packagesToScan, false);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. Useful for tests where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext() {
        boot(false, Collections.<String>emptyList(), false);
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. You can also specify extra packages to scan for classes with XL Deploy annotations. Useful for tests
     * where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext(List<String> packagesToScan) {
        boot(false, packagesToScan, false);
    }

    public static void rebootWithoutGlobalContext(List<String> packagesToScan) {
        unboot();
        boot(false, packagesToScan, true);
    }

    public static void reboot() {
        reboot(null, null);
    }

    public static void reboot(File propertiesFile, GlobalContextManager globalContextManager) {
        DescriptorRegistry.withWriteLock(() -> {
            unboot();
            boot(true, Collections.<String>emptyList(), true, propertiesFile, globalContextManager);
            return null;
        });
    }

    private synchronized static void unboot() {
        isBooted.set(false);
        initialized.set(false);
        DescriptorRegistry.remove(LocalDescriptorRegistry.LOCAL_ID);
        DelegateRegistry.unboot();
    }

    private static void boot(boolean withGlobalContext, List<String> packagesToScan, Boolean forceScan) {
        boot(withGlobalContext, packagesToScan, forceScan, null, null);
    }

    private synchronized static void boot(boolean withGlobalContext,
                                          List<String> packagesToScan,
                                          Boolean forceScan,
                                          File propertiesFile,
                                          GlobalContextManager globalContextManager) {
        if (!isBooted.getAndSet(true)) {
            DescriptorRegistry.withWriteLock( () -> {
                try {
                    TypeDefinitions typeDefinitions = bootStage1(packagesToScan, forceScan);
                    if (withGlobalContext) {
                        if (propertiesFile != null) {
                            GlobalContextInitializer.init(LocalDescriptorRegistry.LOCAL_ID, propertiesFile, globalContextManager);
                        } else {
                            GlobalContextInitializer.init(LocalDescriptorRegistry.LOCAL_ID);
                        }
                    }
                    bootStage2(typeDefinitions);
                } catch (IllegalStateException e) {
                    unboot(); // clear local booter state on failure
                    throw e;
                }
                return null;
            });
        }

        if (!initialized.get()) {
            throw new IllegalStateException("The DescriptorRegistry has been booted, but is not initialized. Please check the logs for any errors.");
        }
    }

    private synchronized static TypeDefinitions bootStage1(List<String> packagesToScan, Boolean forceScan) {
        LocalDescriptorRegistry registry = new LocalDescriptorRegistry();
        DescriptorRegistry.add(registry);

        Scannit scannit = getScannit(packagesToScan, forceScan);

        ValidationRuleConverter.initialize();
        VerificationConverter.initialize();

        DelegateRegistry.boot(scannit);

        final TypeSystemBootstrapper typeSystemBootstrapper = new TypeSystemBootstrapper(registry, scannit);
        TypeDefinitions typeDefinitions = typeSystemBootstrapper.loadTypes();

        registry.verifyTypes();

        return typeDefinitions;
    }

    private synchronized static void bootStage2(TypeDefinitions typeDefinitions) {
        PluginVersions.init();
        ConfigurationItemPostProcessors.registerPostProcessors();
        CiRoots.findRoots(typeDefinitions);
        initialized.set(true);
    }

    private static Scannit getScannit(List<String> packagesToScan, Boolean forceScan) {
        Configuration configuration = Configuration.config()
                .scan("ai.digital")
                .scan("com.xebialabs")
                .scan("ext.deployit") // Deployit Extensions
                .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner());

        for (String pkg : packagesToScan) {
            logger.debug("Scanning additional package '{}' for types, rules etc.", pkg);
            configuration.scan(pkg);
        }

        return Scannit.boot(configuration, forceScan);
    }

    public static boolean isInitialized() {
        return initialized.get();
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalBooter.class);
}
