package com.xebialabs.deployit.booter.local;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.IDescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlplatform.synthetic.BaseTypeSpecification;
import com.xebialabs.xlplatform.synthetic.MethodSpecification;
import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkNotNull;
import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkState;

public class TypeDefinitions {

    private final Map<Type, TypeDefinition> typeDefs = new HashMap<>();
    private IDescriptorRegistry descriptorRegistry;

    public TypeDefinitions(IDescriptorRegistry descriptorRegistry, Collection<TypeDefinition> existingTypeDefinitions) {
        this.descriptorRegistry = descriptorRegistry;
        existingTypeDefinitions.forEach(this::addTypeDef);
        TypeDefinitionsRegistry.register(descriptorRegistry.getId(), this);
    }

    public TypeDefinitions(IDescriptorRegistry descriptorRegistry) {
        this(descriptorRegistry, Collections.emptyList());
    }

    public void addTypeDef(TypeDefinition typeDef) {
        checkState(!typeDefs.containsKey(typeDef.type), "Trying to register duplicate definition for type [%s]. Existing: [%s], duplicate: [%s]", typeDef.type, typeDefs.get(typeDef.type), typeDef);
        typeDefs.put(typeDef.type, typeDef);
    }

    public Collection<TypeDefinition> getDefinitions() {
        return new ArrayList<>(typeDefs.values());
    }

    private TypeDefinition getDefinition(Type type) {
        return checkNotNull(typeDefs.get(type), "Could not find a type definition associated with type [%s]", type);
    }

    public void defineType(Class clazz) {
        @SuppressWarnings("unchecked")
        ClassBasedTypeDefinition typeDef = new ClassBasedTypeDefinition(descriptorRegistry, clazz);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    public void defineType(TypeSpecification typeSpec) {
        addTypeDef(new SyntheticBasedTypeDefinition(descriptorRegistry, typeSpec));

        if (typeSpec.hasGenerateDeployable()) {
            var ownerType = typeSpec.getTypeName().toType(descriptorRegistry);
            addTypeDef(new GenerateDeployableTypeDefinition(ownerType, typeSpec.getGenerateDeployable()));
        }

        findAllGeneratedParameterTypes(typeSpec);
    }

    public void redefineType(TypeSpecification typeSpec) {
        SyntheticBasedTypeDefinition typeDef = new SyntheticBasedTypeDefinition(descriptorRegistry, typeSpec);
        typeDefs.put(typeDef.type, typeDef);

        if (typeSpec.hasGenerateDeployable()) {
            var ownerType = typeSpec.getTypeName().toType(descriptorRegistry);
            GenerateDeployableTypeDefinition generatedTypeDef = new GenerateDeployableTypeDefinition(ownerType, typeSpec.getGenerateDeployable());
            typeDefs.put(generatedTypeDef.type, generatedTypeDef);
        }

        findAllGeneratedParameterTypes(typeSpec);
    }

    void modifyType(TypeModificationSpecification typeModification) {
        var modifiedType = typeModification.getTypeName().toType(descriptorRegistry);
        checkState(typeDefs.containsKey(modifiedType), "Detected type-modification for non-existing type [%s]", modifiedType);

        typeDefs.get(modifiedType).addTypeModification(typeModification);

        findAllGeneratedParameterTypes(typeModification);

        logger.debug("Found: {}", typeModification);
    }

    private void findAllGeneratedParameterTypes(BaseTypeSpecification type) {
        for (MethodSpecification method : type.getMethods()) {
            if (!method.getParameters().isEmpty()) {
                defineGeneratedParameterType(method, type);
            }
        }
    }

    void defineGeneratedParameterType(MethodSpecification methodDef, BaseTypeSpecification baseType) {
        GeneratedParameterTypeDefinition typeDef = new GeneratedParameterTypeDefinition(descriptorRegistry, baseType, methodDef);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    //
    // Registering types
    //

    public void registerTypes() {
        // Create type tree
        getDefinitions().forEach(definition -> registerAsSubtype(definition, definition.type));

        // Register type definitions
        getDefinitions().forEach(this::register);
    }


    private void registerAsSubtype(TypeDefinition definition, Type type) {
        if (definition.superType != null) {
            descriptorRegistry.registerSubtype(definition.superType, type);

            registerAsSubtype(getDefinition(definition.superType), type);
        }
        for (Type anInterface : definition.getInterfaces()) {
            descriptorRegistry.registerSubtype(anInterface, type);
        }
    }

    private void register(TypeDefinition definition) {
        if (descriptorRegistry.exists(definition.type)) {
            return;
        }

        // Register supertype
        if (definition.superType != null && !descriptorRegistry.exists(definition.superType)) {
            register(getDefinition(definition.superType));
        }

        // Register interfaces
        for (Type anInterface : definition.getInterfaces()) {
            if (!descriptorRegistry.exists(anInterface)) {
                register(getDefinition(anInterface));
            }
        }

        // Register owner of a derived type
        if (definition.getOwner() != null && !descriptorRegistry.exists(definition.getOwner())) {
            register(getDefinition(definition.getOwner()));
        }

        // Register type itself
        LocalDescriptor descriptor = definition.createDescriptor();

        // Type modifications
        definition.applyTypeModifications(descriptor);
    }

    public IDescriptorRegistry getDescriptorRegistry() {
        return descriptorRegistry;
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitions.class);
}
