package com.xebialabs.deployit.booter.local

import com.xebialabs.xlplatform.synthetic.TypeDefinitionDocuments.{TYPE_DEFINITIONS_XML, TYPE_DEFINITIONS_YAML}
import com.xebialabs.xlplatform.synthetic.TypeSpecification
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument
import com.xebialabs.xlplatform.synthetic.yaml.TypeDefinitionYamlDocument

import java.io.File
import java.nio.charset.StandardCharsets
import java.util.zip.{ZipEntry, ZipFile}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try, Using}

class PluginsInZipExtension {

  def extractFrom(file: File): Option[Try[List[TypeSpecification]]] = {
    val result = Using(new ZipFile(file)) { zipFile =>
      val typesXml = Option(zipFile.getEntry(TYPE_DEFINITIONS_XML))
      val typesYaml = Option(zipFile.getEntry(TYPE_DEFINITIONS_YAML))

      if (typesXml.isEmpty && typesYaml.isEmpty) {
        Failure(new IllegalArgumentException(s"Plugin doesn't have any type-definitions file"))
      } else if (typesXml.isDefined && typesYaml.isDefined) {
        Failure(new IllegalArgumentException(s"Plugin contains both $TYPE_DEFINITIONS_XML and $TYPE_DEFINITIONS_YAML files"))
      } else {
        val xmlTypeSpecifications = typesXml match {
          case Some(xml) => loadZipEntryContents(zipFile, xml) match {
            case Success(contents) => readXmlDefinition(contents)
            case Failure(ex) => Failure(ex)
          }
          case None => Success(List.empty[TypeSpecification])
        }

        val yamlTypeSpecifications = typesYaml match {
          case Some(yaml) => loadZipEntryContents(zipFile, yaml) match {
            case Success(contents) => readYamlDefinition(contents)
            case Failure(ex) => Failure(ex)
          }
          case None => Success(List.empty[TypeSpecification])
        }

        for {
          xmlTypes <- xmlTypeSpecifications
          yamlTypes <- yamlTypeSpecifications
        } yield xmlTypes ++ yamlTypes
      }
    }
    // Ensure None is never returned because that is considered a valid plugin
    result match {
      case Success(value) => Some(value)
      case Failure(ex) => Some(Failure(ex))
    }
  }

  private def loadZipEntryContents(file: ZipFile, entry: ZipEntry): Try[String] = {
    Using(file.getInputStream(entry)) { stream =>
      new String(stream.readAllBytes(), StandardCharsets.UTF_8)
    }
  }

  private def readXmlDefinition(contents: String): Try[List[TypeSpecification]] = {
    Try(SyntheticXmlDocument.read(contents).getTypes.asScala.toList)
  }

  private def readYamlDefinition(contents: String): Try[List[TypeSpecification]] = {
    Try(TypeDefinitionYamlDocument.read(contents).getTypes.asScala.toList)
  }
}