package com.xebialabs.deployit.booter.local;

import java.lang.reflect.Method;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.creator.CreatorContext;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Delegate;
import com.xebialabs.deployit.plugin.api.udm.Parameters;

import nl.javadude.scannit.Scannit;

import static java.lang.reflect.Modifier.isStatic;

public class DelegateRegistry {
    private static final Map<String, Method> CONTROL_TASK_DELEGATES = new HashMap<>();

    private static final Map<String, Method> CREATOR_DELEGATES = new HashMap<>();

    private static final Logger logger = LoggerFactory.getLogger(DelegateRegistry.class);

    static void boot(Scannit scannit) {
        logger.info("Registering delegates.");

        Set<Method> allDelegateMethods = scannit.getMethodsAnnotatedWith(Delegate.class);
        Set<Method> delegateMethods = getMethods(allDelegateMethods);

        for (Method delegateMethod : delegateMethods) {
            Delegate annotation = delegateMethod.getAnnotation(Delegate.class);
            switch (annotation.use()) {
                case CONTROL_TASK:
                    doRegister(CONTROL_TASK_DELEGATES, annotation.name(), delegateMethod, "Found 2 methods with same control task delegate name [%s]: %s and %s");
                    break;
                case CREATOR:
                    doRegister(CREATOR_DELEGATES, annotation.name(), delegateMethod, "Found 2 methods with same creator delegate name [%s]: %s and %s");
                    break;
            }
            logger.debug("Registered delegate [{}] ({}).", annotation.name(), delegateMethod);
        }
    }

    static void unboot() {
        CONTROL_TASK_DELEGATES.clear();
        CREATOR_DELEGATES.clear();
    }

    private static void doRegister(Map<String, Method> delegate, String name, Method delegateMethod, String s) {
        Method old = delegate.put(name, delegateMethod);
        if (old != null) {
            throw new IllegalStateException(String.format(s, name, old, delegateMethod));
        }
    }

    static Set<Method> getMethods(Set<Method> allDelegateMethods) {
        Set<String> classes = new HashSet<>();
        for (Method m : allDelegateMethods) {
            classes.add(m.getDeclaringClass().getName());
        }
        Set<Method> delegateMethods = new HashSet<>();
        for (Method m : allDelegateMethods) {
            String n = m.getDeclaringClass().getName();
            if (!(n.endsWith("$") && classes.contains(n.substring(0, n.length() - 1)))) {
                delegateMethods.add(m);
            }
        }
        return delegateMethods;
    }

    static void verify(Verifications verifications) {
        for (Map.Entry<String, Method> entry : CONTROL_TASK_DELEGATES.entrySet()) {
            Method method = entry.getValue();
            String name = entry.getKey();
            verifications.verify(isStatic(method.getModifiers()), "Delegate [%s (%s)] should be static", name, method);
            verifications.verify(List.class.isAssignableFrom(method.getReturnType()), "Delegate [%s (%s)] should have a List<Step> return type", name, method);
            Class<?>[] parameterTypes = method.getParameterTypes();
            verifications.verify(parameterTypes.length >= 3 && parameterTypes.length <= 4, "Delegate [%s (%s)] should take 3 or 4 arguments (Taking: %s)", name, method, Arrays.asList(parameterTypes));
            verifications.verify(ConfigurationItem.class.isAssignableFrom(parameterTypes[0]), "Delegate [%s (%s)] should take a ConfigurationItem as first parameter", name, method);
            verifications.verify(parameterTypes[1].equals(String.class), "Delegate [%s (%s)] should take a String as second parameter", name, method);
            verifications.verify(Map.class.isAssignableFrom(parameterTypes[2]), "Delegate [%s (%s)] should take a Map<String, String> as third parameter", name, method);
            if (parameterTypes.length == 4) {
                verifications.verify(Parameters.class.isAssignableFrom(parameterTypes[3]), "Delegate [%s (%s)] should take a ConfigurationItem as fourth parameter", name, method);
            }
        }
        for (Map.Entry<String, Method> entry : CREATOR_DELEGATES.entrySet()) {
            Method method = entry.getValue();
            String name = entry.getKey();
            verifications.verify(isStatic(method.getModifiers()), "CreatorDelegate [%s (%s)] must be static.", name, method);
            verifications.verify(method.getReturnType().equals(Void.TYPE), "CreatorDelegate [%s (%s)] must be declared to return void.", name, method);
            Class<?>[] parameterTypes = method.getParameterTypes();
            verifications.verify(parameterTypes.length >= 1 && parameterTypes.length <= 2, "CreatorDelegate [%s (%s)] must take 1 or 2 arguments (Taking: %s)", name, method, Arrays.asList(parameterTypes));
            verifications.verify(CreatorContext.class.isAssignableFrom(parameterTypes[0]), "CreatorDelegate [%s (%s)] must take a CreatorContext as first parameter", name, method);
            if (parameterTypes.length == 2) {
                verifications.verify(Map.class.isAssignableFrom(parameterTypes[1]), "CreatorDelegate [%s (%s)] must take a Map<String, String> as second parameter", name, method);
            }
        }
    }

    static boolean exists(String name) {
        return exists(name, Delegate.Use.CONTROL_TASK);
    }

    static boolean exists(String name, Delegate.Use use) {
        switch (use) {
            case CONTROL_TASK:
                return CONTROL_TASK_DELEGATES.containsKey(name);
            case CREATOR:
                return CREATOR_DELEGATES.containsKey(name);
        }
        throw new IllegalArgumentException("use: " + use);
    }

    static Method getDelegate(String name) {
        return getDelegate(name, Delegate.Use.CONTROL_TASK);
    }

    static Method getDelegate(String name, Delegate.Use use) {
        switch (use) {
            case CONTROL_TASK:
                return CONTROL_TASK_DELEGATES.get(name);
            case CREATOR:
                return CREATOR_DELEGATES.get(name);
        }
        throw new IllegalArgumentException("use: " + use);
    }

    // Non-static delegates are deprecated
    static Object instantiateDelegate(String name) {
        try {
            return CONTROL_TASK_DELEGATES.get(name).getDeclaringClass().newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            throw new RuntimeException("Could not instantiate delegate: " + name, e);
        }
    }

}
