package com.xebialabs.deployit.booter.local

import java.util

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.booter.local.utils.CheckUtils.checkArgument
import javax.xml.bind.DatatypeConverter

import scala.jdk.CollectionConverters._

private[local] trait Converter {
  def convert(s: String): Any
}

private[local] object Converter {
  def splitValue(s: String): Seq[String] = s.split(",").toIndexedSeq.map(_.trim).filter(_.nonEmpty)
}

private[local] class NullCheckConverter(notNull: (String) => Any) extends Converter {
  override def convert(s: String): Any = Option(s).map(notNull).orNull
}

private[local] object Converters {

  def createStatelessConverter(name: String, kind: PropertyKind): Converter = kind match {
    case BOOLEAN => booleanConverter
    case INTEGER => integerConverter
    case STRING => stringConverter
    case ENUM => throw new IllegalArgumentException("ENUM converter is not stateless")
    case DATE => dateConverter
    case SET_OF_STRING => setOfStringConverter
    case LIST_OF_STRING => listOfStringConverter
    case MAP_STRING_STRING => new MapStringStringConverter(name, kind)
    case _ =>
      throw new IllegalArgumentException("Property " + name + " of kind " + kind + " cannot be converted from a string value")
  }

  val booleanConverter: Converter = new NullCheckConverter(java.lang.Boolean.parseBoolean)
  val integerConverter: Converter = new NullCheckConverter(s => if (s.isEmpty) null else s.toInt)
  val stringConverter: Converter = new NullCheckConverter(identity)
  val dateConverter: Converter = new NullCheckConverter(DatatypeConverter.parseDateTime(_).getTime)
  val setOfStringConverter: Converter = new NullCheckConverter(s => Converter.splitValue(s).toSet.asJava)
  val listOfStringConverter: Converter = new NullCheckConverter(s => Converter.splitValue(s).toList.asJava)

}

private[local] class MapStringStringConverter(name: String, kind: PropertyKind) extends NullCheckConverter(value => {
  Converter.splitValue(value).map(_.split(":")).foldLeft(new util.LinkedHashMap[String, String]()){(map, split) =>
    checkArgument(Strings.defaultIfEmpty(split(0), null) != null,
            "Property '%s' of kind '%s' cannot be converted from string value '%s' because of any empty key.", name, kind, value)
    map.put(split(0), if (split.length == 1) "" else split(1))
    map
  }
})

private[local] class EnumClassConverter(enumClass: Class[_]) extends NullCheckConverter(s => {
  enumClass.getEnumConstants.asInstanceOf[Array[Enum[_ <: Enum[_]]]]
    .find(_.name().equalsIgnoreCase(s))
    .getOrElse(throw new IllegalArgumentException(s"Value '$s' not a member of enum '$enumClass'"))
})

private[local] class EnumValuesConverter(enumValues: util.List[String]) extends NullCheckConverter(s => {
  enumValues.asScala.find(_.equalsIgnoreCase(s))
    .getOrElse(throw new IllegalArgumentException(s"Value '$s' not a member of possible enum values"))
})



