package com.xebialabs.deployit.booter.local;

import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.*;

public class GlobalContextRegistry {

    final private static Logger logger = LoggerFactory.getLogger(GlobalContextRegistry.class);

    private static ConcurrentHashMap<DescriptorRegistryId, GlobalContext> contexts = new ConcurrentHashMap<>();

    public static GlobalContext getGlobalContext(DescriptorRegistryId registryId) {
        return contexts.computeIfAbsent(registryId, id -> {
            var gc = new GlobalContext();
            DescriptorRegistry.getDescriptorRegistry(registryId).registerHook(new DescriptorRegistryHook() {
                @Override
                public void onClose(final IDescriptorRegistry registry) {
                    logger.debug("Removing GlobalContext that belongs to registry {} from GlobalContextRegistry", registry.getId());
                    contexts.remove(registry.getId());
                }
            });
            return gc;
        });
    }

    public static GlobalContext unregister(DescriptorRegistryId registryId) {
        return contexts.remove(registryId);
    }

    private static IDescriptorRegistry getDescriptorRegistry(final PropertyDescriptor pd) {
        final DescriptorRegistryId typeSource = pd.getTypeSource();
        IDescriptorRegistry registry = DescriptorRegistry.getDescriptorRegistry(typeSource);
        return registry;
    }

    private static GlobalContext getGlobalContext(PropertyDescriptor pd) {
        var descriptorRegistry = getDescriptorRegistry(pd);
        var descriptorRegistryId = descriptorRegistry.getId();
        return getGlobalContext(descriptorRegistryId);
    }

    static String lookup(PropertyDescriptor propertyDescriptor) {
        var globalContext = getGlobalContext(propertyDescriptor);
        return globalContext.lookup(propertyDescriptor);
    }

    static void register(PropertyDescriptor pd, String defaultValue) {
        var globalContext = getGlobalContext(pd);
        globalContext.register(pd, defaultValue);
    }

    static void register(PropertyDescriptor pd, PropertyDescriptor inheritedFrom) {
        var globalContext = getGlobalContext(pd);
        globalContext.register(pd, inheritedFrom);
    }

    public static void copy(DescriptorRegistryId from, DescriptorRegistryId to) {
        var sourceGlobalContext = GlobalContextRegistry.getGlobalContext(from);
        // initializer our global context with values from parent?
        var targetGlobalContext = GlobalContextRegistry.getGlobalContext(to);
        targetGlobalContext.copyFrom(sourceGlobalContext);
    }
}
