package com.xebialabs.deployit.booter.local;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.generator.TypeGenerators;
import com.xebialabs.deployit.plugin.api.reflect.IDescriptorRegistry;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument;
import com.xebialabs.xlplatform.synthetic.yaml.TypeDefinitionYamlDocument;

import nl.javadude.scannit.Scannit;

import static com.xebialabs.xlplatform.synthetic.TypeDefinitionDocuments.SYNTHETIC_TEST_XML;
import static com.xebialabs.xlplatform.synthetic.TypeDefinitionDocuments.SYNTHETIC_XML;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static java.io.File.separator;
import static java.util.Collections.list;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;
import static org.apache.commons.lang.StringUtils.EMPTY;

class TypeSystemBootstrapper {

    private static final String PLUGINS = separator + "plugins" + separator;

    private IDescriptorRegistry registry;
    private final Scannit scannit;
    private final TypeDefinitions typeDefinitions;
    private final TypeGenerators typeGenerators;

    TypeSystemBootstrapper(IDescriptorRegistry registry, Scannit scannit) {
        this.registry = registry;
        this.scannit = scannit;
        this.typeDefinitions = new TypeDefinitions(registry);
        this.typeGenerators = new TypeGenerators(scannit);
    }

    TypeDefinitions loadTypes() {
        logger.info("Initializing type system.");

        scanClasses();
        scanSynthetics();

        typeDefinitions.registerTypes();
        typeGenerators.generateTypeDefinitions(typeDefinitions);
        typeDefinitions.registerTypes();

        return typeDefinitions;
    }

    //
    // Class-based types
    //

    private void scanClasses() {
        typeDefinitions.defineType(ConfigurationItem.class);
        Set<Class<? extends ConfigurationItem>> cis = scannit.getSubTypesOf(ConfigurationItem.class);
        for (Class<? extends ConfigurationItem> ci : cis) {
            typeDefinitions.defineType(ci);
        }
    }

    //
    // Synthetic types
    //

    private void scanSynthetics() {
        try {
            List<TypeSpecification> types = new ArrayList<>();
            List<TypeModificationSpecification> typeModifications = new ArrayList<>();
            readSyntheticXml(SYNTHETIC_XML, types, typeModifications);
            readSyntheticXml(SYNTHETIC_TEST_XML, types, typeModifications);

            types.forEach(typeDefinitions::defineType);
            typeModifications.forEach(typeDefinitions::modifyType);
        } catch (IOException ex) {
            throw new RuntimeException("Could not read synthetic type definitions.", ex);
        }
    }

    private void readSyntheticXml(String name, List<TypeSpecification> types, List<TypeModificationSpecification> typeModifications) throws IOException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);

        for (URL syntheticXML : sortedResources(syntheticXMLs)) {
            logger.debug("Scanning synthetic XML: {}", syntheticXML);

            SyntheticXmlDocument syntheticXml = SyntheticXmlDocument.read(registry, syntheticXML);

            types.addAll(syntheticXml.getTypes());
            typeModifications.addAll(syntheticXml.getTypeModifications());
        }
    }

    private void readSyntheticYaml(String name, List<TypeSpecification> types) throws IOException {
        Enumeration<URL> files = MODULE$.classLoader().getResources(name);

        for (URL yamlFile : sortedResources(files)) {
            logger.debug("Scanning YAML type defintions: {}", yamlFile);

            TypeDefinitionYamlDocument typeDefinition = TypeDefinitionYamlDocument.read(registry, yamlFile);

            types.addAll(typeDefinition.getTypes());
        }
    }

    private List<URL> sortedResources(Enumeration<URL> syntheticXMLs) {
        List<URL> files = list(syntheticXMLs);

        return concat(allButPlugins(files), sortedPlugins(files))
                .collect(toList());
    }

    private Stream<URL> allButPlugins(List<URL> files) {
        return files.stream()
                .filter(filterByFolder(PLUGINS).negate());
    }

    private Predicate<URL> filterByFolder(String s) {
        return p -> decode(p).contains(s);
    }

    private String decode(URL url) {
        String decodedString = EMPTY;
        try {
            decodedString = URLDecoder.decode(url.getPath(), "UTF-8");
        } catch (UnsupportedEncodingException ignored) {
        }
        return decodedString;
    }

    private Stream<URL> sortedPlugins(List<URL> files) {
        return files.stream()
                .filter(filterByFolder(PLUGINS))
                .sorted(this::sortByPluginFolderName);
    }

    private int sortByPluginFolderName(URL url1, URL url2) {
        return getPluginFolderName(url1).compareTo(getPluginFolderName(url2));
    }

    private String getPluginFolderName(URL url) {
        String path = decode(url);
        return path.substring(path.lastIndexOf(PLUGINS) + PLUGINS.length());
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeSystemBootstrapper.class);
}
