package com.xebialabs.deployit.booter.local;

import java.io.File;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;

final class GlobalContext {
    private final Map<String, Value> context = new ConcurrentHashMap<>();

    private final AtomicReference<GlobalContextManager> globalContextManagerReference = new AtomicReference<>(
            new LocalGlobalContextManager().withContext(context)
    );

    public static final File DEFAULTS = new File("conf", "deployit-defaults.properties");

    GlobalContext() {
    }

    void setManager(GlobalContextManager globalContextManager) {
        globalContextManagerReference.set(
                globalContextManager.withContext(context)
        );
    }

    GlobalContextManager getManager() {
        return globalContextManagerReference.get();
    }

    void register(PropertyDescriptor pd, String defaultValue) {
        if (defaultValue == null && !pd.isHidden()) return;
        context.put(pd.getFqn(), new Value(pd.getDescription(), defaultValue));
    }


    void register(PropertyDescriptor pd, PropertyDescriptor inheritedFrom) {
        context.put(pd.getFqn(), new InheritedValue(this, pd.getDescription(), inheritedFrom.getFqn(), pd.isHidden()));
    }

    String lookup(PropertyDescriptor pd) {
        return lookup(pd.getFqn());
    }

    String lookup(String key) {
        Value value = context.get(key);
        if (value == null) return null;
        return value.getValue();
    }

    void copyFrom(GlobalContext other) {
        // copy values from another global context into this one
        var copiedEntries = other.context.entrySet().stream()
                .collect(
                        Collectors.toMap(e -> e.getKey(), e -> e.getValue().copyInto(this))
                );
        this.context.putAll(copiedEntries);
    }

    void init(File defaultFile) {
        getManager().loadStoredDefaults(defaultFile);
        getManager().validateValues(defaultFile);
        getManager().storeDefaults(defaultFile);
    }

    static class InheritedValue extends Value {
        private GlobalContext context;
        private final String superTypeProperty;
        private final boolean shouldWrite;

        InheritedValue(GlobalContext context, String description, String superTypeProperty, boolean shouldWrite) {
            super(description, "");
            this.context = context;
            this.superTypeProperty = superTypeProperty;
            this.shouldWrite = shouldWrite;
        }

        @Override
        public String getValue() {
            return context.lookup(superTypeProperty);
        }

        @Override
        boolean isShouldWrite() {
            return shouldWrite;
        }

        String getSuperTypeProperty() {
            return superTypeProperty;
        }

        Value copyInto(GlobalContext targetContext) {
            return new InheritedValue(targetContext, description, superTypeProperty, shouldWrite);
        }

    }

    static class Value {
        Value(String description, String value) {
            this.description = description;
            this.value = value;
        }

        final String description;
        final String value;

        boolean isExplicit() {
            return false;
        }

        boolean isShouldWrite() {
            return true;
        }

        public String getValue() {
            return value;
        }

        Value copyInto(GlobalContext context) {
            return new Value(description, value);
        }
    }

    static class ExplicitValue extends Value {
        ExplicitValue(String description, String value) {
            super(description, value);
        }

        @Override
        boolean isExplicit() {
            return true;
        }

        Value copyInto(GlobalContext context) {
            return new ExplicitValue(description, value);
        }

    }
}
