package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

import com.xebialabs.deployit.plugin.api.creator.CreatorContext;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.EmbeddedDeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.EmbeddedDeployedArtifact;
import com.xebialabs.deployit.plugin.api.validation.Validator;

import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkArgument;
import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkNotNull;
import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkState;
import static com.xebialabs.deployit.booter.local.utils.ClassUtils.getActualTypeArguments;
import static com.xebialabs.deployit.booter.local.utils.Strings.defaultIfEmpty;

class ClassBasedLocalDescriptor extends LocalDescriptor {
    private boolean isInterface;
    private transient IDescriptorRegistry descriptorRegistry;

    ClassBasedLocalDescriptor(IDescriptorRegistry localDescriptorRegistry, Class<? extends ConfigurationItem> clazz) {
        this.descriptorRegistry = localDescriptorRegistry;
        this.isInterface = clazz.isInterface();
        setType(typeOf(clazz));
        setVirtual(clazz.isInterface());
        setClazz(clazz);
        initMetadata();
        scanClass();
        initHierarchy();
        // As a last one, once we have all PropertyDescriptors, init our default field values
        initFieldDefaults();
        resolveIcon();
    }

    private void initMetadata() {
        if (isInterface) {
            return;
        }

        Class<? extends ConfigurationItem> clazz = getClazz();

        boolean directlyAnnotated = false;
        Annotation[] declaredAnnotations = clazz.getDeclaredAnnotations();
        for (Annotation declaredAnnotation : declaredAnnotations) {
            if (declaredAnnotation.annotationType().equals(Metadata.class)) {
                directlyAnnotated = true;
            }
        }

        Metadata annotation = getMetadataAnnotation();
        setLabel(defaultIfEmpty(annotation.label(), toLabel(getType())));
        setDescription(annotation.description());
        if(annotation.root() == Metadata.ConfigurationItemRoot.BY_ROOT_NAME) {
            setRootName(Optional.of(annotation.rootName()));
        } else {
            setRootName(Optional.ofNullable(annotation.root().getRootNodeName()));
        }
        setInspectable(annotation.inspectable());
        setVirtual((directlyAnnotated && annotation.virtual()) || Modifier.isAbstract(clazz.getModifiers()));
        setVersioned(annotation.versioned());
    }

    private Metadata getMetadataAnnotation() {
        Class<? extends ConfigurationItem> clazz = getClazz();
        return checkNotNull(clazz.getAnnotation(Metadata.class), "Class " + clazz.getName() + " or one of its ancestors does not have a @Metadata annotation");
    }

    private void scanClass() {
        findProperties();
        findControlTasks();
        findInterfaces();
        findValidations();
        findVerifications();
        findCreator();

        Class<?> superclass = getClazz().getSuperclass();
        if (superclass != null && ConfigurationItem.class.isAssignableFrom(superclass)) {
            Type supertype = typeOf(superclass);
            addSuperClass(supertype);
        }

        initDeployableAndContainerTypes();
    }

    private void initFieldDefaults() {
        if (Modifier.isAbstract(getClazz().getModifiers()) || !hasDefaultConstructor()) {
            return;
        }

        ConfigurationItem configurationItem = newCleanInstance();

        for (PropertyDescriptor propertyDescriptor : getPropertyDescriptors()) {
            if (Arrays.asList(PropertyKind.CI, PropertyKind.SET_OF_CI, PropertyKind.LIST_OF_CI).contains(propertyDescriptor.getKind())) {
                continue;
            }
            if (propertyDescriptor.getDefaultValue() == null) {
                Object fieldValue = propertyDescriptor.get(configurationItem);
                if (fieldValue instanceof Collection && ((Collection) fieldValue).isEmpty()) {
                    // Skip
                } else if (fieldValue instanceof Map && ((Map) fieldValue).isEmpty()) {
                    // Skip
                } else if (fieldValue != null) {
                    GlobalContext.register(propertyDescriptor, fieldValue.toString());
                }
            }
        }
    }

    private boolean hasDefaultConstructor() {
        try {
            getClazz().getConstructor();
            return true;
        } catch (NoSuchMethodException nsme) {
            return false;
        }
    }

    private void findVerifications() {
        for (Annotation annotation : getClazz().getAnnotations()) {
            if (VerificationConverter.isVerification(annotation)) {
                this.verifications.add(VerificationConverter.makeVerification(annotation));
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void findValidations() {
        for (Annotation annotation : getClazz().getAnnotations()) {
            if (ValidationRuleConverter.isRule(annotation)) {
                Validator<ConfigurationItem> v = (Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(annotation, getType());
                this.validators.add(v);
            }
        }
    }

    private void findControlTasks() {
        for (Method method : getClazz().getDeclaredMethods()) {
            if (method.isAnnotationPresent(ControlTask.class)) {
                MethodDescriptor from = LocalMethodDescriptor.from(this, method);
                addControlTask(from);
            }
        }
    }

    private void findCreator() {
        boolean isSet = false;
        for (Method method : getClazz().getDeclaredMethods()) {
            if (method.isAnnotationPresent(Creator.class)) {
                checkState(!isSet, "Found 2 creators for class [%s]. Only one allowed.", getClazz());
                checkState(method.getReturnType().equals(Void.TYPE), "Creator [%s] of class [%s] must be declared to return void.", method.getName(), getClazz());
                checkState(method.getParameterCount() <= 2, "Creator [%s] of class [%s] has more than 2 parameters. Only 2 allowed (CreatorContext and Map)", method.getName(), getClazz());
                checkState(method.getParameterCount() > 0, "Creator [%s] of class [%s] must take at least one argument (CreatorContext).", method.getName(), getClazz());
                checkState(Modifier.isStatic(method.getModifiers()), "Creator [%s] of class [%s] must be static.", method.getName(), getClazz());
                checkState(CreatorContext.class.isAssignableFrom(method.getParameterTypes()[0]), "Creator [%s] of class [%s] must take a CreatorContext as the first parameter.", method.getName(), getClazz());
                if (method.getParameterCount() == 2) {
                    checkState(Map.class.isAssignableFrom(method.getParameterTypes()[1]), "Creator [%s] of class [%s] must take a Map<String, String> as the second parameter.", method.getName(), getClazz());
                }
                setCreator(LocalCreatorDescriptor.from(this, method));
                isSet = true;
            }
        }
    }

    private void findProperties() {
        for (Field field : getClazz().getDeclaredFields()) {
            if (field.isAnnotationPresent(Property.class)) {
                LocalPropertyDescriptor propertyDescriptor = new FieldBasedPropertyDescriptor(this, field);
                addPropertyDescriptor(propertyDescriptor);
            }
        }
    }

    private void findInterfaces() {
        Class<?>[] clazzInterfaces = getClazz().getInterfaces();
        List<Class<?>> allInterfacesFound = new ArrayList<>();
        findAllSuperInterfaces(clazzInterfaces, allInterfacesFound);
        for (Class<?> clazzInterface : allInterfacesFound) {
            if (clazzInterface.getPackage().isAnnotationPresent(Prefix.class) && ConfigurationItem.class.isAssignableFrom(clazzInterface)) {
                addInterface(typeOf(clazzInterface));
            }
        }
    }

    private void findAllSuperInterfaces(Class<?>[] childInterfaces, List<Class<?>> allInterfacesFound) {
        for (Class<?> childInterface : childInterfaces) {
            allInterfacesFound.add(childInterface);
            findAllSuperInterfaces(childInterface.getInterfaces(), allInterfacesFound);
        }
    }

    private void initDeployableAndContainerTypes() {
        Class<? extends ConfigurationItem> clazz = getClazz();
        if (Deployed.class.isAssignableFrom(clazz)) {
            initDeployableAndContainerTypes(Deployed.class, Deployable.class, Container.class);
        } else if (EmbeddedDeployed.class.isAssignableFrom(clazz) && !DerivedArtifact.class.isAssignableFrom(clazz)) {
            initDeployableAndContainerTypes(EmbeddedDeployed.class, EmbeddedDeployable.class, EmbeddedDeployedContainer.class);
        } else if (EmbeddedDeployedArtifact.class.isAssignableFrom(clazz) && DerivedArtifact.class.isAssignableFrom(clazz)) {
            initDeployableAndContainerTypes(EmbeddedDeployedArtifact.class, EmbeddedDeployableArtifact.class, EmbeddedDeployedContainer.class);
        }
    }

    private void initDeployableAndContainerTypes(Class<? extends ConfigurationItem> deployedClass, Class<? extends ConfigurationItem> deployableClass, Class<? extends ConfigurationItem> containerClass) {
        @SuppressWarnings("unchecked")
        List<Class<?>> typeArguments = getActualTypeArguments(getClazz().asSubclass(deployedClass), (Class<ConfigurationItem>) deployedClass);
        checkArgument(typeArguments.size() == 2, "Expected a %s and a %s, but got %s", deployableClass.getSimpleName(), containerClass.getSimpleName(), typeArguments);

        Class<?> genericDeployableClass = typeArguments.get(0);
        if (genericDeployableClass != null) {
            checkArgument(deployableClass.isAssignableFrom(genericDeployableClass), "Expected first item to be a %s", deployableClass.getSimpleName());
            setDeployableType(typeOf(genericDeployableClass));
        }

        Class<?> genericContainerClass = typeArguments.get(1);
        if (genericContainerClass != null) {
            checkArgument(containerClass.isAssignableFrom(genericContainerClass), "Expected second item to be a %s", containerClass.getSimpleName());
            setContainerType(typeOf(genericContainerClass));
        }
    }

    boolean isInterface() {
        return isInterface;
    }

    @Override
    protected IDescriptorRegistry descriptorRegistry() {
        return descriptorRegistry;
    }
}
