package com.xebialabs.deployit.booter.local;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.IDescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.synthetic.BaseTypeSpecification;
import com.xebialabs.xlplatform.synthetic.MethodSpecification;
import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkNotNull;
import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkState;
import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.getAllInterfaces;
import static java.util.stream.Collectors.toList;

public class TypeDefinitions {

    private final Map<Type, TypeDefinition> typeDefs = new HashMap<>();
    private IDescriptorRegistry registry;

    public TypeDefinitions(IDescriptorRegistry registry, Collection<TypeDefinition> existingTypeDefinitions) {
        this.registry = registry;
        existingTypeDefinitions.forEach(this::addTypeDef);
        TypeDefinitionsRegistry.register(registry.getId(), this);
    }

    public TypeDefinitions(IDescriptorRegistry registry) {
        this(registry, Collections.emptyList());
    }

    public void addTypeDef(TypeDefinition typeDef) {
        checkState(!typeDefs.containsKey(typeDef.type), "Trying to register duplicate definition for type [%s]. Existing: [%s], duplicate: [%s]", typeDef.type, typeDefs.get(typeDef.type), typeDef);
        typeDefs.put(typeDef.type, typeDef);
    }

    public Collection<TypeDefinition> getDefinitions() {
        return new ArrayList<>(typeDefs.values());
    }

    private TypeDefinition getDefinition(Type type) {
        return checkNotNull(typeDefs.get(type), "Could not find a type definition associated with type [%s]", type);
    }

    public void defineType(Class clazz) {
        @SuppressWarnings("unchecked")
        ClassBasedTypeDefinition typeDef = new ClassBasedTypeDefinition(registry, clazz);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    public void defineType(TypeSpecification typeSpec) {
        addTypeDef(new SyntheticBasedTypeDefinition(typeSpec));

        if (typeSpec.hasGenerateDeployable()) {
            addTypeDef(new GenerateDeployableTypeDefinition(typeSpec.getType(), typeSpec.getGenerateDeployable()));
        }

        findAllGeneratedParameterTypes(typeSpec);
    }

    public void redefineType(TypeSpecification typeSpec) {
        SyntheticBasedTypeDefinition typeDef = new SyntheticBasedTypeDefinition(typeSpec);
        typeDefs.put(typeDef.type, typeDef);

        if (typeSpec.hasGenerateDeployable()) {
            GenerateDeployableTypeDefinition generatedTypeDef = new GenerateDeployableTypeDefinition(typeSpec.getType(), typeSpec.getGenerateDeployable());
            typeDefs.put(generatedTypeDef.type, generatedTypeDef);
        }

        findAllGeneratedParameterTypes(typeSpec);
    }

    void modifyType(TypeModificationSpecification typeModification) {
        checkState(typeDefs.containsKey(typeModification.getType()), "Detected type-modification for non-existing type [%s]", typeModification.getType());

        typeDefs.get(typeModification.getType()).addTypeModification(typeModification);

        findAllGeneratedParameterTypes(typeModification);

        logger.debug("Found: {}", typeModification);
    }

    private void findAllGeneratedParameterTypes(BaseTypeSpecification type) {
        for (MethodSpecification method : type.getMethods()) {
            if (!method.getParameters().isEmpty()) {
                defineGeneratedParameterType(method, type);
            }
        }
    }

    void defineGeneratedParameterType(MethodSpecification methodDef, BaseTypeSpecification baseType) {
        GeneratedParameterTypeDefinition typeDef = new GeneratedParameterTypeDefinition(registry, baseType, methodDef);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    //
    // Registering types
    //

    public void registerTypes() {
        // Create type tree
        getDefinitions().forEach(definition -> registerAsSubtype(definition, definition.type));

        // Register type definitions
        getDefinitions().forEach(this::register);
    }


    private void registerAsSubtype(TypeDefinition definition, Type type) {
        if (definition.superType != null) {
            registry.registerSubtype(definition.superType, type);

            registerAsSubtype(getDefinition(definition.superType), type);
        }
        for (Type anInterface : definition.getInterfaces()) {
            registry.registerSubtype(anInterface, type);
        }
    }

    private void register(TypeDefinition definition) {
        if (registry._exists(definition.type)) {
            return;
        }

        // Register supertype
        if (definition.superType != null && !registry._exists(definition.superType)) {
            register(getDefinition(definition.superType));
        }

        // Register interfaces
        for (Type anInterface : definition.getInterfaces()) {
            // TODO S-88045 LocalDescriptorRegistry.exists might a problem
            //  maybe Deploy which has remote descriptor registries does something different?
            if (!registry._exists(anInterface)) {
                register(getDefinition(anInterface));
            }
        }

        // Register owner of a derived type
        if (definition.getOwner() != null && !registry._exists(definition.getOwner())) {
            register(getDefinition(definition.getOwner()));
        }

        // Register type itself
        LocalDescriptor descriptor = definition.createDescriptor();
        registry.register(descriptor);

        // Type modifications
        definition.applyTypeModifications(descriptor);
    }

    //
    // Helper methods
    //

    private static boolean isClassACi(Class<?> clazz) {
        return clazz != null && ConfigurationItem.class.isAssignableFrom(clazz);
    }


    //
    // Helper classes
    //

    class ClassBasedTypeDefinition extends TypeDefinition {
        private Class<? extends ConfigurationItem> clazz;
        private IDescriptorRegistry registry;

        public ClassBasedTypeDefinition(IDescriptorRegistry registry, Class<? extends ConfigurationItem> ci) {
            this.registry = registry;
            this.type = type(ci);
            this.superType = type(ci.getSuperclass());
            this.interfaces = getAllInterfaces(ci).stream().map(this::type).filter(Objects::nonNull).collect(toList());
            this.clazz = ci;
        }

        @Override
        protected LocalDescriptor createDescriptor() {
            return new ClassBasedLocalDescriptor(this.registry, clazz);
        }

        private Type type(Class<?> clazz) {
            if (isClassACi(clazz)) {
                return registry.lookupType(clazz);
            } else {
                return null;
            }
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitions.class);
}
