package com.xebialabs.xlplatform.synthetic.xml;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.function.Predicate;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.xebialabs.xlplatform.synthetic.TypeName;

interface XmlOperations {

    default String getRequiredStringAttribute(Element element, String attributeName) {
        if (element.hasAttribute(attributeName)) {
            return element.getAttribute(attributeName);
        }
        throw new IllegalArgumentException("Attribute " + attributeName + " not provided");
    }

    default TypeName getRequiredTypeAttribute(Element element, String attributeName) {
        TypeName type = getOptionalTypeAttribute(element, attributeName);
        if (type == null) {
            throw new IllegalArgumentException("Attribute " + attributeName + " not provided on element " + element.getNodeName());
        }
        return type;
    }

    default TypeName getOptionalTypeAttribute(Element element, String attributeName, TypeName defaultValue) {
        String typeAttr = getOptionalStringAttribute(element, attributeName, null);
        if (typeAttr != null) {
            return new TypeName(typeAttr);
        } else {
            return defaultValue;
        }
    }

    default TypeName getOptionalTypeAttribute(Element element, String attributeName) {
        return getOptionalTypeAttribute(element, attributeName, null);
    }

    default Optional<TypeName> getOptionalType(Element element, String attributeName) {
        return getOptionalString(element, attributeName).map(TypeName::new);
    }

    default boolean getOptionalBooleanAttribute(Element element, String attributeName, boolean defaultValue) {
        return getOptionalBoolean(element, attributeName).orElse(defaultValue);
    }

    default Optional<Boolean> getOptionalBoolean(Element element, String attributeName) {
        return getOptionalString(element, attributeName).map(Boolean::valueOf);
    }

    default String getOptionalStringAttribute(Element element, String attributeName, String defaultValue) {
        return getOptionalString(element, attributeName).orElse(defaultValue);
    }

    default Optional<String> getOptionalString(Element element, String attributeName) {
        if (element.hasAttribute(attributeName)) {
            return Optional.of(element.getAttribute(attributeName));
        } else {
            return Optional.empty();
        }
    }

    default Optional<String> getOptionalTextOfChild(Element element, String childElementName) {
        Iterator<Element> children = childrenByName(element, childElementName::equals).iterator();
        if (children.hasNext()) {
            return Optional.ofNullable(children.next().getFirstChild().getTextContent());
        }
        return Optional.empty();
    }

    default Iterable<Element> childrenByName(final Element element, final Predicate<String> matcher) {
        return () -> childByName(element, matcher);
    }

    default Iterator<Element> childByName(final Element element, final Predicate<String> matcher) {
        return new Iterator<Element>() {
            private int i = -1;
            private final NodeList childNodes = element.getChildNodes();
            private int nextIndex = -1;

            @Override
            public boolean hasNext() {
                if (nextIndex == i) nextIndex = findNext();
                return nextIndex > i;
            }

            @Override
            public Element next() {
                if (nextIndex == i) nextIndex = findNext();
                if (nextIndex < i) throw new NoSuchElementException("There are no more matching elements");
                i = nextIndex;

                return (Element) childNodes.item(i);
            }

            private int findNext() {
                int next = i;
                while (next < childNodes.getLength()) {
                    next++;
                    if (childNodes.item(next) instanceof Element) {
                        Element e = (Element) childNodes.item(next);
                        if (matcher.test(e.getNodeName())) {
                            return next;
                        }
                    }
                }
                return i - 1;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    default <E> void forEach(Iterator<E> iterator, Closure<E> closure) {
        while (iterator.hasNext()) {
            closure.call(iterator.next());
        }
    }

}
