package com.xebialabs.xlplatform.synthetic.yaml;

import com.fasterxml.jackson.databind.JsonNode;
import com.xebialabs.xlplatform.synthetic.InputHintSpecification;
import com.xebialabs.xlplatform.synthetic.PropertySpecification;

import java.util.*;

public class JsonPropertySpecification extends JsonSpecification implements PropertySpecification {

    protected String defaultCategory = null;
    protected boolean defaultHidden = false;

    public JsonPropertySpecification(String name, JsonNode node) {
        super(name, node);
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Optional<String> getKind() {
        String defaultKind = null;
        if (node.has("referenced-type")) {
            defaultKind = "ci";
        }
        return getOptionalString("kind", defaultKind);
    }

    @Override
    public Optional<String> getCategory() {
        return getOptionalString("category", defaultCategory);
    }

    @Override
    public Optional<Boolean> getRequired() {
        // Note: required is false by default on purpose. This is different in synthetic.xml
        return getOptionalBoolean("required");
    }

    @Override
    public Optional<Boolean> getPassword() {
        return getOptionalBoolean("password");
    }

    @Override
    public Optional<Boolean> getAsContainment() {
        return getOptionalBoolean("as-containment");
    }

    @Override
    public Optional<Boolean> getNested() {
        return getOptionalBoolean("nested");
    }

    @Override
    public Optional<String> getSize() {
        return getOptionalString("size");
    }

    @Override
    public Optional<String> getDefault() {

        // Shortcut so you can write
        //   my-property: "Default value"
        if (node.isTextual()) {
            return Optional.of(node.asText());
        }

        if (node.has("default")) {
            JsonNode value = node.get("default");

            if (value.isValueNode()) {
                return Optional.of(value.asText());
            } else if (value.isArray()) {
                // Convert Yaml lists to the expected String format
                return Optional.of(toSyntheticListFormat(value));
            } else if (value.isObject()) {
                // Convert Yaml objects to the expected String format for maps
                return Optional.of(toSyntheticMapFormat(value));
            }
        }

        return Optional.empty();
    }

    private static String toSyntheticListFormat(JsonNode value) {
        StringBuilder builder = new StringBuilder();
        boolean first = true;
        for (JsonNode item : value) {
            if (!first) {
                builder.append(",");
            }
            first = false;

            builder.append(item.asText());
        }
        return builder.toString();
    }

    private static String toSyntheticMapFormat(JsonNode node) {
        StringBuilder builder = new StringBuilder();
        boolean first = true;

        Iterator<String> keys = node.fieldNames();
        while (keys.hasNext()) {
            if (!first) {
                builder.append(",");
            }
            first = false;

            String key = keys.next();
            String value = node.get(key).asText();
            builder.append(key).append(":").append(value);
        }
        return builder.toString();
    }

    @Override
    public Optional<Boolean> getHidden() {
        return getOptionalBoolean("hidden", defaultHidden);
    }

    @Override
    public Optional<Boolean> getInspectionProperty() {
        return getOptionalBoolean("inspection-property");
    }

    @Override
    public Set<String> getAliases() {
        return new LinkedHashSet<>(getStringValues("aliases"));
    }

    @Override
    public Optional<Boolean> getTransient() {
        return getOptionalBoolean("transient");
    }

    @Override
    public Optional<String> getCandidateValuesFilter() {
        return getOptionalString("candidate-values-filter");
    }

    @Override
    public Optional<Boolean> getReadOnly() {
        return getOptionalBoolean("readonly");
    }

    @Override
    public Optional<String> getEnumClass() {
        return getOptionalString("enum-class");
    }

    @Override
    public List<String> getEnumValues() {
        return getStringValues("enum-values");
    }

    @Override
    public Optional<String> getReferencedType() {
        return getOptionalString("referenced-type");

    }

    @Override
    public List<InputHintSpecification> getInputHints() {
        if (!node.has("input-hint")) {
            return Collections.emptyList();
        }

        // Synthetic.xsd only allows one input-hint
        JsonInputHintSpecification hint = new JsonInputHintSpecification(node.get("input-hint"));
        return Collections.singletonList(hint);
    }

}
