package com.xebialabs.deployit.booter.local;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.*;

import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkArgument;
import static com.xebialabs.deployit.booter.local.utils.CheckUtils.checkState;


public class LocalDescriptorRegistry extends BaseDescriptorRegistry {
    public static final int ORDER = IDescriptorRegistry.DEFAULT_PRECEDENCE - 1;
    private static Logger logger = LoggerFactory.getLogger(LocalDescriptorRegistry.class);

    public static final DescriptorRegistryId LOCAL_ID = new LocalDescriptorRegistryId();

    private final Map<Type, Descriptor> descriptors = new HashMap<>();

    private final SealedDescriptorHierarchy subtypes = new SealedDescriptorHierarchy();

    protected LocalDescriptorRegistry() {
        super(LOCAL_ID);
    }

    public void register(Descriptor descriptor) {
        checkState(!this.descriptors.containsKey(descriptor.getType()), "The type [%s] is already registered.", descriptor.getType());
        logger.debug("Registered ConfigurationItem: {}", descriptor);

        this.descriptors.put(descriptor.getType(), descriptor);
    }

    public void registerSubtype(Type supertype, Type subtype) {
        checkState(!supertype.equals(subtype), "Cannot register [%s] as its own subtype.", supertype);
        this.subtypes.registerSubType(supertype, subtype);
    }

    @Override
    public boolean isLocal() {
        return true;
    }

    @Override
    public boolean isDefault() {
        return true;
    }

    @Override
    public int getOrder() {
        return ORDER;
    }

    @Override
    public Collection<Descriptor> _getDescriptors() {
        return Collections.unmodifiableCollection(descriptors.values());
    }

    @Override
    public Collection<Type> _getSubtypes(Type supertype) {
        return Collections.unmodifiableCollection(subtypes.getSubtypes(supertype));
    }

    @Override
    public Descriptor _getDescriptor(Type type) {
        checkState(!descriptors.isEmpty(), "DescriptorRegistry not booted");
        checkArgument(_exists(type), "Unknown type [%s]", type);
        return descriptors.get(type);
    }

    @Override
    public boolean _exists(Type type) {
        return descriptors.containsKey(type);
    }

    @Override
    public void verifyTypes() {
        Verifications verifications = new Verifications();
        DelegateRegistry.verify(verifications);
        _getDescriptors().stream().filter(input -> !input.isVirtual()).forEach(d -> ((LocalDescriptor) d).verify(verifications));
        verifications.done();
    }
}

