package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.plugin.api.reflect.InputHint;
import com.xebialabs.deployit.plugin.api.reflect.InputHintValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static com.xebialabs.deployit.booter.local.utils.Strings.deCamelize;
import static com.xebialabs.deployit.booter.local.utils.Strings.split;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.*;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static com.xebialabs.deployit.plugin.api.reflect.InputHintValue.inputHintValue;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils.classByName;

class SyntheticLocalPropertyDescriptor extends LocalPropertyDescriptor {


    public SyntheticLocalPropertyDescriptor(LocalDescriptor descriptor, Element propertyElement) {
        String name = getRequiredStringAttribute(propertyElement, "name");
        setName(name);
        setDeclaringDescriptor(descriptor);
        initSynthetic(propertyElement);
    }

    private void initSynthetic(Element propertyElement) {
        PropertyKind pk = PropertyKind.valueOf(getOptionalStringAttribute(propertyElement, "kind", STRING.name()).toUpperCase());
        if (pk == PropertyKind.BOOLEAN) {
            setPrimitiveKind(pk);
        } else {
            setKind(pk);
        }
        setCategory(getOptionalStringAttribute(propertyElement, "category", "Common"));
        setLabel(getOptionalStringAttribute(propertyElement, "label", deCamelize(getName())));
        setDescription(getOptionalStringAttribute(propertyElement, "description", getLabel()));
        setRequired(getOptionalBooleanAttribute(propertyElement, "required", true));
        reInitializeRequired();
        setPassword(getOptionalBooleanAttribute(propertyElement, "password", false));
        setAsContainment(getOptionalBooleanAttribute(propertyElement, "as-containment", false));
        setSize(Property.Size.valueOf(getOptionalStringAttribute(propertyElement, "size", Property.Size.DEFAULT.name()).toUpperCase()));
        String defaultValueAttr = getOptionalStringAttribute(propertyElement, "default", null);
        setHidden(getOptionalBooleanAttribute(propertyElement, "hidden", false));
        setInspectionProperty(getOptionalBooleanAttribute(propertyElement, "inspectionProperty", false));
        setAliases(new HashSet<>(split(getOptionalStringAttribute(propertyElement, "aliases", ""), ",")));
        setTransient(isHidden() || getOptionalBooleanAttribute(propertyElement, "transient", false));
        setCandidateValuesFilter(getOptionalStringAttribute(propertyElement, "candidate-values-filter", null));

        PropertyKind kind = getKind();

        if (getKind() == ENUM) {
            if (propertyElement.hasAttribute("enum-class")) {
                String enumClassAttributeValue = getRequiredStringAttribute(propertyElement, "enum-class", "for property " + getName() + " of kind " + getKind());
                try {
                    Class<?> enumClass = classByName(enumClassAttributeValue);
                    setEnumClass(enumClass);
                    if (!enumClass.isEnum()) {
                        throw new IllegalArgumentException("enum-class supplied for property " + getName() + " of kind " + getKind() + " is not an enum: "
                                + enumClassAttributeValue);
                    }
                    initEnumValues(enumClass);
                } catch (ClassNotFoundException exc) {
                    throw new IllegalArgumentException("Unknown enum-class supplied for property " + getName() + " of kind " + getKind() + ": " + enumClassAttributeValue,
                            exc);
                }
            } else {
                final NodeList evs = propertyElement.getElementsByTagName("enum-values");
                List<String> enumValues = new ArrayList<>();
                if (evs.getLength() == 1) {
                    forEach(childByName((Element) evs.item(0), "value"::equals), element -> enumValues.add(element.getTextContent()));
                } else {
                    throw new IllegalArgumentException("Could not find 'enum-class' attribute or 'enum-values' element defined on property " + getName() + " of kind " + getKind());
                }
                setEnumValues(enumValues);
            }
        }

        if ( kind == CI || kind == SET_OF_CI || kind == LIST_OF_CI) {
            setReferencedType(Type.valueOf(getRequiredStringAttribute(propertyElement, "referenced-type", "for property " + getName() + " of kind " + getKind())));
        }

        initSyntheticValidationRules(propertyElement);
        addDefaultValidationRules();
        initSyntheticInputHints(propertyElement, kind);
        registerDefault(defaultValueAttr);
    }

    private void initSyntheticInputHints(final Element propertyElement, PropertyKind propertyKind) {
        forEach(childByName(propertyElement, "input-hint"::equals), inputHintElement -> {
            InputHint inputHint = new InputHint();
            inputHint.setKind(propertyKind);
            forEach(childByName(inputHintElement, "values"::equals), valuesElement -> {
                List<InputHintValue> inputHintValues = new ArrayList<>();
                forEach(childByName(valuesElement, "value"::equals), valueElement -> {
                    String value = valueElement.getTextContent();
                    String label = getOptionalStringAttribute(valueElement, "label", value);
                    inputHintValues.add(inputHintValue(value, label));
                });
                inputHint.setValues(inputHintValues);
            });
            forEach(childByName(inputHintElement, "rule"::equals), rule ->
                    inputHint.getValidationRules().add(ValidationRuleConverter.makeRule(rule, SyntheticLocalPropertyDescriptor.this)));
            forEach(childByName(inputHintElement, "prompt"::equals), prompt ->
                    inputHint.setPrompt(prompt.getFirstChild().getTextContent()));
            forEach(childByName(inputHintElement, "copy-from-property"::equals), copyFromProperty ->
                inputHint.setCopyFromProperty(copyFromProperty.getFirstChild().getTextContent()));
            inputHint.setRequired(isRequired());
            setInputHint(inputHint);
        });
    }

    private void initSyntheticValidationRules(Element propertyElement) {
        forEach(childByName(propertyElement, "rule"::equals), element ->
                validationRules.add(ValidationRuleConverter.makeRule(element, SyntheticLocalPropertyDescriptor.this)));
    }

    @Override
    public Object get(ConfigurationItem item) {
        return getDeclaringDescriptor().getSyntheticPropertyValue(item, getName());
    }

    @Override
    protected void doSetValue(ConfigurationItem item, Object value) {
        getDeclaringDescriptor().setSyntheticPropertyValue(item, getName(), value);
    }
}
