package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.reflect.TypeVerification;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.validation.Validator;
import org.w3c.dom.Element;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import static com.google.common.base.Predicates.equalTo;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.*;

class SyntheticParser {

    final static SyntheticParser PARSER = new SyntheticParser();

    Type parseType(Element element) {
        return getRequiredTypeAttribute(element, "type");
    }

    Type parseSuperType(Element element) {
        return getRequiredTypeAttribute(element, "extends");
    }

    Optional<String> parseLabel(Element typeElement) {
        return getOptionalString(typeElement, "label");
    }

    Optional<String> parseDescription(Element typeElement) {
        return getOptionalString(typeElement, "description");
    }

    Optional<Boolean> parseVirtual(Element typeElement) {
        return getOptionalBoolean(typeElement, "virtual");
    }

    Optional<Boolean> parseInspectable(Element typeElement) {
        return getOptionalBoolean(typeElement, "inspectable");
    }

    Optional<String> parseIcon(Element typeElement) {
        Iterator<Element> generateElements = childByName(typeElement, equalTo("icon"));
        if (generateElements.hasNext()) {
            Element iconElement = generateElements.next();
            return Optional.ofNullable(iconElement.getTextContent());
        } else {
            return Optional.empty();
        }
    }

    Optional<String> parseRoot(Element typeElement) {
        return getOptionalString(typeElement, "root").flatMap(rootName -> {
            for (Metadata.ConfigurationItemRoot configurationItemRoot : Metadata.ConfigurationItemRoot.values()) {
                if (rootName.equalsIgnoreCase(configurationItemRoot.getRootNodeName())) {
                    if (configurationItemRoot == Metadata.ConfigurationItemRoot.BY_ROOT_NAME) {
                        return Optional.of(getRequiredStringAttribute(typeElement, "rootName"));
                    }
                    return Optional.of(configurationItemRoot.getRootNodeName());
                }
            }
            return Optional.empty();
        });
    }

    List<TypeVerification> parseVerifications(final Element typeElement, Type type) {
        List<TypeVerification> verifications = new ArrayList<>();
        forEach(childByName(typeElement, equalTo(VerificationConverter.VERIFICATION_ELEMENT_NAME)),
                element -> verifications.add(VerificationConverter.makeVerification(element, type)));
        return verifications;
    }

    List<Validator<ConfigurationItem>> parseValidators(Element typeElement, Type type) {
        List<Validator<ConfigurationItem>> validators = new ArrayList<>();
        //noinspection unchecked
        forEach(childByName(typeElement, equalTo(ValidationRuleConverter.RULE_ELEMENT_NAME)),
                element -> validators.add((Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(element, type)));
        return validators;
    }

    Optional<Type> parseDeployedType(Element element) {
        return getOptionalType(element, "deployable-type");
    }

    Optional<Type> parseConteinerType(Element element) {
        return getOptionalType(element, "container-type");
    }

    GeneratedDeployableDescription parseGeneratedDeployableDescription(Element typeElement) {
        Iterator<Element> generateElements = childByName(typeElement, equalTo("generate-deployable"));
        if (generateElements.hasNext()) {
            Element generateDeployable = generateElements.next();
            return new GeneratedDeployableDescription(
                    getRequiredTypeAttribute(generateDeployable, "type"),
                    getRequiredTypeAttribute(generateDeployable, "extends"),
                    getOptionalBooleanAttribute(generateDeployable, "copy-default-values", false),
                    getOptionalBooleanAttribute(generateDeployable, "virtual", false),
                    getOptionalStringAttribute(generateDeployable, "description", "Description unavailable")
            );
        }
        return null;
    }


    Optional<Boolean> parseVersioned(final Element typeModElement) {
        return getOptionalBoolean(typeModElement, "versioned");
    }

    List<MethodDescriptor> parseControlTasks(LocalDescriptor descriptor, final Element element) {
        List<MethodDescriptor> methodDescriptors = new ArrayList<>();
        forEach(childByName(element, equalTo("method")), element1 -> {
            MethodDescriptor controlTask = LocalMethodDescriptor.from(descriptor, element1);
            methodDescriptors.add(controlTask);
        });
        return methodDescriptors;
    }

    List<LocalPropertyDescriptor> parseProperties(LocalDescriptor descriptor, Element element) {
        List<LocalPropertyDescriptor> propertyDescriptors = new ArrayList<>();
        forEach(childByName(element, equalTo("property")), element1 -> {
            getRequiredStringAttribute(element1, "name");
            LocalPropertyDescriptor newDesc = new SyntheticLocalPropertyDescriptor(descriptor, element1);
            propertyDescriptors.add(newDesc);
        });
        return propertyDescriptors;
    }
}
