package com.xebialabs.deployit.booter.local.utils;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import com.google.common.base.Predicate;

import com.xebialabs.deployit.plugin.api.reflect.Type;

public class XmlUtils {
    public static String getRequiredStringAttribute(Element element, String attributeName) {
        if (element.hasAttribute(attributeName)) {
            return element.getAttribute(attributeName);
        }
        throw new IllegalArgumentException("Attribute " + attributeName + " not provided");
    }

    public static String getRequiredStringAttribute(Element element, String attributeName, String additionalMessage) {
        if (element.hasAttribute(attributeName)) {
            return element.getAttribute(attributeName);
        }
        throw new IllegalArgumentException("Attribute " + attributeName + " not provided " + additionalMessage);
    }

    public static Type getRequiredTypeAttribute(Element element, String attributeName) {
        Type type = getOptionalTypeAttribute(element, attributeName);
        if (type == null) {
            throw new IllegalArgumentException("Attribute " + attributeName + " not provided");
        }
        return type;
    }

    public static Type getOptionalTypeAttribute(Element element, String attributeName, Type defaultValue) {
        String typeAttr = getOptionalStringAttribute(element, attributeName, null);
        Type type = defaultValue;
        if (typeAttr != null) {
            type = Type.valueOf(typeAttr);
        }
        return type;
    }

    public static Type getOptionalTypeAttribute(Element element, String attributeName) {
        return getOptionalTypeAttribute(element, attributeName, null);
    }

    public static Optional<Type> getOptionalType(Element element, String attributeName) {
        return getOptionalString(element, attributeName).map(Type::valueOf);
    }

    public static boolean getOptionalBooleanAttribute(Element element, String attributeName, boolean defaultValue) {
        return getOptionalBoolean(element, attributeName).orElse(defaultValue);
    }

    public static Optional<Boolean> getOptionalBoolean(Element element, String attributeName) {
        return getOptionalString(element, attributeName).map(Boolean::valueOf);
    }

    public static String getOptionalStringAttribute(Element element, String attributeName, String defaultValue) {
        return getOptionalString(element, attributeName).orElse(defaultValue);
    }

    public static Optional<String> getOptionalString(Element element, String attributeName) {
        if (element.hasAttribute(attributeName)) {
            return Optional.ofNullable(element.getAttribute(attributeName));
        } else {
            return Optional.empty();
        }
    }

    public static Iterable<Element> childrenByName(final Element element, final Predicate<String> matcher) {
        return new Iterable<Element>() {
            @Override
            public Iterator<Element> iterator() {
                return childByName(element, matcher);
            }
        };
    }

    public static Iterator<Element> childByName(final Element element, final Predicate<String> matcher) {
        return new Iterator<Element>() {
            private int i = -1;
            private NodeList childNodes = element.getChildNodes();
            private int nextIndex = -1;

            @Override
            public boolean hasNext() {
                if (nextIndex == i) nextIndex = findNext();
                return nextIndex > i;
            }

            @Override
            public Element next() {
                if (nextIndex == i) nextIndex = findNext();
                if (nextIndex < i) throw new NoSuchElementException("There are no more matching elements");
                i = nextIndex;

                return (Element) childNodes.item(i);
            }

            private int findNext() {
                int next = i;
                while (next < childNodes.getLength()) {
                    next++;
                    if (childNodes.item(next) instanceof Element) {
                        Element e = (Element) childNodes.item(next);
                        if (matcher.apply(e.getNodeName())) {
                            return next;
                        }
                    }
                }
                return i - 1;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public static <E> void forEach(Iterator<E> iterator, Closure<E> closure) {
        while (iterator.hasNext()) {
            closure.call(iterator.next());
        }
    }

    public static interface Closure<E> {
        void call(E e);
    }

}
