package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Parameters;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.childByName;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getOptionalStringAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getOptionalTypeAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredStringAttribute;

class LocalMethodDescriptor implements MethodDescriptor {

    static final String CONTROL_TASK_DISPATCH_METHOD = "controlTaskDispatch";
    private static final Set<String> RESERVED_ATTRIBUTE_NAMES = newHashSet("name", "label", "description", "parameters-type");

    private String name;
    private String label;
    private String description;
    private String delegate;
    private Map<String, String> attributes = newHashMap();
    private Descriptor descriptor;
    private List<MethodVerification> verifications = newArrayList();
    private Type parameterType;

    private LocalMethodDescriptor(Descriptor descriptor, String name) {
        this.descriptor = descriptor;
        this.name = name;
    }

    private LocalMethodDescriptor(Descriptor descriptor, Method method) {
        this.name = method.getName();
        this.delegate = "methodInvoker";
        this.descriptor = descriptor;
    }

    LocalMethodDescriptor(LocalMethodDescriptor copyOf, Descriptor newOwner) {
        this.name = copyOf.name;
        this.description = copyOf.description;
        this.label = copyOf.label;
        this.attributes = copyOf.attributes;
        this.delegate = copyOf.delegate;
        this.parameterType = copyOf.parameterType;
        this.descriptor = newOwner;
    }

    static MethodDescriptor from(Descriptor descriptor, Method method) {
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, method);
        methodDescriptor.initMetadata(method);
        initVerifications(methodDescriptor);
        return methodDescriptor;
    }

    static MethodDescriptor from(Descriptor descriptor, Element element) {
        String name = getRequiredStringAttribute(element, "name");
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, name);
        methodDescriptor.label = getOptionalStringAttribute(element, "label", name);
        methodDescriptor.description = getOptionalStringAttribute(element, "description", "No description.");
        methodDescriptor.delegate = getOptionalStringAttribute(element, "delegate", "dispatcherInvoker");
        if (element.hasAttribute("parameters-type")) {
            methodDescriptor.parameterType = getOptionalTypeAttribute(element, "parameters-type");
        } else {
            Iterator<Element> parameters = childByName(element, equalTo("parameters"));
            if (parameters.hasNext()) {
                methodDescriptor.parameterType = TypeDefinitions.generatedParameterType(descriptor.getType(), name);
            }
        }
        methodDescriptor.attributes = readAttributes(element);
        initVerifications(methodDescriptor);
        return methodDescriptor;
    }

    private static void initVerifications(final LocalMethodDescriptor methodDescriptor) {
        Method delegateMethod = checkNotNull(DelegateRegistry.getDelegate(methodDescriptor.delegate), methodDescriptor.delegate + " is referenced, but not registered.");
        for (Annotation annotation : delegateMethod.getAnnotations()) {
            if (VerificationConverter.isVerification(annotation)) {
                methodDescriptor.verifications.add(VerificationConverter.<MethodVerification>makeVerification(annotation));
            }
        }
    }

    private static Map<String, String> readAttributes(final Element element) {
        Map<String, String> map = newHashMap();
        NamedNodeMap xmlAttrs = element.getAttributes();
        for (int i = 0; i < xmlAttrs.getLength(); i++) {
            Attr item = (Attr) xmlAttrs.item(i);
            if (!RESERVED_ATTRIBUTE_NAMES.contains(item.getName())) {
                map.put(item.getName(), item.getValue());
            }
        }

        return map;
    }

    private void initMetadata(Method method) {
        ControlTask annotation = method.getAnnotation(ControlTask.class);
        description = annotation.description();
        label = annotation.label().equals("") ? name : annotation.label();
        parameterType = annotation.parameterType().equals("") ? parameterType : Type.valueOf(annotation.parameterType());
    }

    void verify(final Verifications verifications) {
        verifications.verify(descriptor.getType(), DelegateRegistry.exists(delegate), "No delegate called [%s] available for control task [%s]", delegate, getFqn());
        Type superType = Type.valueOf(Parameters.class);
        verifications.verify(parameterType == null || parameterType.isSubTypeOf(superType), "The parameter type [%s] for control task [%s] should be a subtype of [%s]", parameterType, getFqn(), superType);
        for (MethodVerification verification : this.verifications) {
            verification.verify(this, new VerificationContext() {
                @Override
                public void error(final String message, final Object... params) {
                    verifications.verify(LocalMethodDescriptor.this.descriptor.getType(), false, message, params);
                }
            });
        }
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public String getDescription() {
        return description;
    }

    Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    public Type getParameterObjectType() {
        return parameterType;
    }

    @Override
    public <T> T invoke(ConfigurationItem item) {
        //noinspection RedundantTypeArguments
        return this.<T>invokeDelegate(item, null);
    }

    @Override
    public <T> T invoke(final ConfigurationItem item, final Parameters params) {
        //noinspection RedundantTypeArguments
        return this.<T>invokeDelegate(item, params);
    }

    @SuppressWarnings("unchecked")
    private <T> T invokeDelegate(final ConfigurationItem item, final Parameters params) {
        Method method = DelegateRegistry.getDelegate(delegate);
        Object o = null;

        if (!Modifier.isStatic(method.getModifiers())) {
            Deprecations.deprecated("**Deprecated** Non-static delegates are deprecated (Found while invoking [%s]).", getFqn());
            o = DelegateRegistry.instantiateDelegate(delegate);
        }
        try {
            if (method.getParameterTypes().length == 3) {
                return (T) method.invoke(o, item, name, attributes);
            } else {
                return (T) method.invoke(o, item, name, attributes, params);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + name + " on " + item, e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + name + " on " + item);
        }
    }

    @Override
    public String getFqn() {
        return String.format("%s.%s", descriptor.getType(), name);
    }
}
