package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

import com.xebialabs.xlplatform.synthetic.MethodSpecification;
import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Parameters;

import static com.xebialabs.deployit.plugin.api.udm.ControlTask.DEFAULT_DELEGATE;

import static com.xebialabs.overthere.util.OverthereUtils.checkNotNull;

public class LocalMethodDescriptor implements MethodDescriptor {

    static final String CONTROL_TASK_DISPATCH_METHOD = "controlTaskDispatch";

    private String name;
    private String label;
    private String description;
    private String delegate;
    private Map<String, String> attributes = new HashMap<>();
    private Descriptor descriptor;
    private List<MethodVerification> verifications = new ArrayList<>();
    private Type parameterType;

    private LocalMethodDescriptor(Descriptor descriptor, String name) {
        this.descriptor = descriptor;
        this.name = name;
    }

    private LocalMethodDescriptor(Descriptor descriptor, Method method) {
        this.name = method.getName();
        this.delegate = DEFAULT_DELEGATE;
        this.descriptor = descriptor;
    }

    LocalMethodDescriptor(LocalMethodDescriptor copyOf, Descriptor newOwner) {
        this.name = copyOf.name;
        this.description = copyOf.description;
        this.label = copyOf.label;
        this.attributes = copyOf.attributes;
        this.delegate = copyOf.delegate;
        this.parameterType = copyOf.parameterType;
        this.descriptor = newOwner;
    }

    static MethodDescriptor from(Descriptor descriptor, Method method) {
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, method);
        methodDescriptor.initMetadata(method);
        initVerifications(methodDescriptor);
        return methodDescriptor;
    }

    public static MethodDescriptor from(Descriptor descriptor, MethodSpecification methodSpecification) {
        String name = methodSpecification.getName();
        LocalMethodDescriptor methodDescriptor = new LocalMethodDescriptor(descriptor, name);
        methodDescriptor.label = methodSpecification.getLabel().orElse(name);
        methodDescriptor.description = methodSpecification.getDescription().orElse("No description.");
        methodDescriptor.delegate = methodSpecification.getDelegate().orElse("dispatcherInvoker");
        Type parameterType = methodSpecification.getParameterType();
        if (parameterType != null) {
            methodDescriptor.parameterType = parameterType;
        } else {
            if (!methodSpecification.getParameters().isEmpty()) {
                methodDescriptor.parameterType = TypeDefinitions.generatedParameterType(descriptor.getType(), name);
            }
        }
        methodDescriptor.attributes = methodSpecification.getAttributes();
        initVerifications(methodDescriptor);
        return methodDescriptor;
    }

    private static void initVerifications(final LocalMethodDescriptor methodDescriptor) {
        Method delegateMethod = checkNotNull(DelegateRegistry.getDelegate(methodDescriptor.delegate), methodDescriptor.delegate + " is referenced, but not registered.");
        for (Annotation annotation : delegateMethod.getAnnotations()) {
            if (VerificationConverter.isVerification(annotation)) {
                methodDescriptor.verifications.add(VerificationConverter.<MethodVerification>makeVerification(annotation));
            }
        }
    }

    private void initMetadata(Method method) {
        ControlTask annotation = method.getAnnotation(ControlTask.class);
        description = annotation.description();
        label = annotation.label().equals("") ? name : annotation.label();
        delegate = annotation.delegate();
        parameterType = annotation.parameterType().equals("") ? parameterType : Type.valueOf(annotation.parameterType());
    }

    void verify(final Verifications verifications) {
        verifications.verify(descriptor.getType(), DelegateRegistry.exists(delegate), "No delegate called [%s] available for control task [%s]", delegate, getFqn());
        Type superType = Type.valueOf(Parameters.class);
        verifications.verify(parameterType == null || parameterType.isSubTypeOf(superType), "The parameter type [%s] for control task [%s] should be a subtype of [%s]", parameterType, getFqn(), superType);
        for (MethodVerification verification : this.verifications) {
            verification.verify(this, new VerificationContext() {
                @Override
                public void error(final String message, final Object... params) {
                    verifications.verify(LocalMethodDescriptor.this.descriptor.getType(), false, message, params);
                }
            });
        }
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public String getDescription() {
        return description;
    }

    Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    public Type getParameterObjectType() {
        return parameterType;
    }

    @Override
    public <T> T invoke(ConfigurationItem item) {
        //noinspection RedundantTypeArguments
        return this.<T>invokeDelegate(item, null);
    }

    @Override
    public <T> T invoke(final ConfigurationItem item, final Parameters params) {
        //noinspection RedundantTypeArguments
        return this.<T>invokeDelegate(item, params);
    }

    @SuppressWarnings("unchecked")
    private <T> T invokeDelegate(final ConfigurationItem item, final Parameters params) {
        Method method = DelegateRegistry.getDelegate(delegate);
        Object o = null;

        if (!Modifier.isStatic(method.getModifiers())) {
            Deprecations.deprecated("**Deprecated** Non-static delegates are deprecated (Found while invoking [%s]).", getFqn());
            o = DelegateRegistry.instantiateDelegate(delegate);
        }
        try {
            if (method.getParameterTypes().length == 3) {
                return (T) method.invoke(o, item, name, attributes);
            } else {
                return (T) method.invoke(o, item, name, attributes, params);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + name + " on " + item, e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + name + " on " + item);
        }
    }

    @Override
    public String getFqn() {
        return String.format("%s.%s", descriptor.getType(), name);
    }
}
