package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.creator.CreatorContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodVerification;
import com.xebialabs.deployit.plugin.api.reflect.VerificationContext;
import com.xebialabs.deployit.plugin.api.reflect.Verify;
import com.xebialabs.deployit.plugin.api.udm.*;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.booter.local.LocalMethodDescriptor.CONTROL_TASK_DISPATCH_METHOD;

/**
 * Used through reflection...
 */
@SuppressWarnings("UnusedDeclaration")
public class DefaultDelegates {

    @SuppressWarnings("unchecked")
    @Delegate(name = ControlTask.DEFAULT_DELEGATE)
    public static List<Step> invokeMethod(ConfigurationItem ci, final String methodName, Map<String, String> arguments, Parameters parameters) {
        List<Method> methods = Arrays.asList(ci.getClass().getMethods());
        Method method = methods.stream().filter(m -> m.getName().equals(methodName) && m.isAnnotationPresent(ControlTask.class)).findFirst().get();
        try {
            if (method.getParameterTypes().length == 1) {
                return (List<Step>) method.invoke(ci, parameters);
            } else {
                return (List<Step>) method.invoke(ci);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + methodName + " on " + ci, e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + methodName + " on " + ci);
        }
    }

    @SuppressWarnings("unchecked")
    @Delegate(name = "dispatcherInvoker")
    @DispatcherVerification
    public static List<Step> invokeDispatcher(ConfigurationItem ci, final String methodName, Map<String, String> arguments, Parameters parameters) {
        Method dispatcher = getDispatcher(ci);

        try {
            if (dispatcher.getParameterTypes().length == 3) {
                return (List<Step>) dispatcher.invoke(ci, methodName, arguments, parameters);
            } else if (dispatcher.getParameterTypes().length == 2) {
                return (List<Step>) dispatcher.invoke(ci, methodName, arguments);
            }
            return (List<Step>) dispatcher.invoke(ci, methodName);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + methodName + " on " + ci);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + methodName + " on " + ci, e);
        }
    }

    @Delegate(use = Delegate.Use.CREATOR, name = ControlTask.DEFAULT_DELEGATE)
    public static void invokeMethod(CreatorContext context, Map<String, String> arguments) {
        List<Method> methods = Arrays.asList(context.getThisCI().getClass().getMethods());
        Method method = methods.stream().filter((m) -> m.isAnnotationPresent(Creator.class)).findFirst().get();
        try {
            if (method.getParameterCount() == 2) {
                method.invoke(null, context, arguments);
            } else {
                method.invoke(null, context);
            }
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not invoke " + method.getName() + " on " + context.getThisCI(), e);
        } catch (InvocationTargetException e) {
            throw ReflectionUtils.handleInvocationTargetException(e, "Could not invoke " + method.getName() + " on " + context.getThisCI());
        }
    }

    private static Method getDispatcher(final ConfigurationItem ci) {
        Class<?> clazz = ci.getClass();
        return Arrays.stream(clazz.getMethods()).filter(m -> m.getName().equals(CONTROL_TASK_DISPATCH_METHOD)).findFirst().get();
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    @Verify(clazz = DispatcherVerification.Verification.class, type = "dispatcherVerification")
    @interface DispatcherVerification {
        class Verification implements MethodVerification {
            @Override
            public void verify(final MethodDescriptor descriptor, final VerificationContext context) {
                Class<?> clazz = ((LocalMethodDescriptor) descriptor).getDescriptor().getClazz();
                List<Method> filter = Arrays.stream(clazz.getMethods()).filter(m -> m.getName().equals(CONTROL_TASK_DISPATCH_METHOD)).collect(Collectors.toList());
                if (filter.size() == 0) {
                    context.error("ControlTask dispatcher [%s] for [%s] is not present.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                } else if (filter.size() > 1) {
                    context.error("Found more than 1 ControlTask dispatcher [%s] for [%s].", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                } else {
                    Method method = filter.get(0);
                    Class<?>[] parameterTypes = method.getParameterTypes();

                    boolean methodWithArgumentsAndParameters = parameterTypes.length == 3;
                    boolean methodWithOnlyArguments = parameterTypes.length == 2;

                    if (!List.class.isAssignableFrom(method.getReturnType())) {
                        context.error("ControlTask dispatcher [%s] for [%s] should return a List<Step>", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (!String.class.isAssignableFrom(parameterTypes[0])) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take a String as first parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if ((methodWithArgumentsAndParameters || methodWithOnlyArguments) && !Map.class.isAssignableFrom(parameterTypes[1])) {
                        context.error("ControlTask dispatcher [%s] for [%s] should take a Map<String,String> as second parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (methodWithArgumentsAndParameters && !Parameters.class.isAssignableFrom(parameterTypes[2])) {
                        context.error("ControlTask dispatcher [%s] for [%s] should take a [udm.Parameters] sub-type as third parameter.", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }

                    if (descriptor.getParameterObjectType() != null && !methodWithArgumentsAndParameters) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take parameters of type [%s].", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn(), descriptor.getParameterObjectType());
                    }

                    if (descriptor.getParameterObjectType() == null && descriptor.getAttributes().size() > 0 && !methodWithOnlyArguments) {
                        context.error("ControlTask dispatcher [%s] for [%s] doesn't take a Map of arguments", CONTROL_TASK_DISPATCH_METHOD, descriptor.getFqn());
                    }
                }
            }
        }
    }

}
