package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.xlplatform.synthetic.BaseTypeSpecification;
import com.xebialabs.xlplatform.synthetic.MethodSpecification;
import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.getAllInterfaces;
import static com.xebialabs.overthere.util.OverthereUtils.checkNotNull;
import static com.xebialabs.overthere.util.OverthereUtils.checkState;
import static java.util.stream.Collectors.toList;

public class TypeDefinitions {

    private final Map<Type, TypeDefinition> typeDefs = new HashMap<>();
    private LocalDescriptorRegistry registry;

    public TypeDefinitions(LocalDescriptorRegistry registry) {
        this.registry = registry;
    }

    public void addTypeDef(TypeDefinition typeDef) {
        checkState(!typeDefs.containsKey(typeDef.type), "Trying to register duplicate definition for type [%s]. Existing: [%s], duplicate: [%s]", typeDef.type, typeDefs.get(typeDef.type), typeDef);
        typeDefs.put(typeDef.type, typeDef);
    }

    public Collection<TypeDefinition> getDefinitions() {
        return new ArrayList<>(typeDefs.values());
    }

    private TypeDefinition getDefinition(Type type) {
        return checkNotNull(typeDefs.get(type), "Could not find a type definition associated with type [%s]", type);
    }

    void defineType(Class clazz) {
        @SuppressWarnings("unchecked")
        ClassBasedTypeDefinition typeDef = new ClassBasedTypeDefinition(clazz);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    void defineType(TypeSpecification typeSpec) {
        addTypeDef(new SyntheticBasedTypeDefinition(typeSpec));

        if (typeSpec.hasGenerateDeployable()) {
            addTypeDef(new GenerateDeployableTypeDefinition(typeSpec.getType(), typeSpec.getGenerateDeployable()));
        }

        findAllGeneratedParameterTypes(typeSpec);
    }

    void modifyType(TypeModificationSpecification typeModification) {
        checkState(typeDefs.containsKey(typeModification.getType()), "Detected type-modification for non-existing type [%s]", typeModification.getType());

        typeDefs.get(typeModification.getType()).addTypeModification(typeModification);

        findAllGeneratedParameterTypes(typeModification);

        logger.debug("Found: {}", typeModification);
    }

    private void findAllGeneratedParameterTypes(BaseTypeSpecification type) {
        for (MethodSpecification method : type.getMethods()) {
            if (!method.getParameters().isEmpty()) {
                defineGeneratedParameterType(method, type);
            }
        }
    }

    void defineGeneratedParameterType(MethodSpecification methodDef, BaseTypeSpecification type) {
        GeneratedParameterTypeDefinition typeDef = new GeneratedParameterTypeDefinition(type, methodDef);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    //
    // Registering types
    //

    public void registerTypes() {
        // Create type tree
        getDefinitions().forEach(definition -> registerAsSubtype(definition, definition.type));

        // Register type definitions
        getDefinitions().forEach(this::register);
    }


    private void registerAsSubtype(TypeDefinition definition, Type type) {
        if (definition.superType != null) {
            registry.registerSubtype(definition.superType, type);

            registerAsSubtype(getDefinition(definition.superType), type);
        }
        for (Type anInterface : definition.getInterfaces()) {
            registry.registerSubtype(anInterface, type);
        }
    }

    private void register(TypeDefinition definition) {
        if (DescriptorRegistry.exists(definition.type)) {
            return;
        }

        // Register supertype
        if (definition.superType != null && !DescriptorRegistry.exists(definition.superType)) {
            register(getDefinition(definition.superType));
        }

        // Register interfaces
        for (Type anInterface : definition.getInterfaces()) {
            if (!LocalDescriptorRegistry.exists(anInterface)) {
                register(getDefinition(anInterface));
            }
        }

        // Register owner of a derived type
        if (definition.getOwner() != null && !DescriptorRegistry.exists(definition.getOwner())) {
            register(getDefinition(definition.getOwner()));
        }

        // Register type itself
        LocalDescriptor descriptor = definition.createDescriptor(this);
        registry.register(descriptor);

        // Type modifications
        definition.applyTypeModifications(descriptor);
    }

    //
    // Helper methods
    //

    private static boolean isClassACi(Class<?> clazz) {
        return clazz != null && ConfigurationItem.class.isAssignableFrom(clazz);
    }

    private static Type type(Class<?> clazz) {
        if (isClassACi(clazz)) {
            return Type.valueOf(clazz);
        } else {
            return null;
        }
    }

    public static Type generatedParameterType(final Type type, final String name) {
        return Type.valueOf(type.toString() + "_" + name);
    }

    //
    // Helper classes
    //

    class ClassBasedTypeDefinition extends TypeDefinition {
        private Class<? extends ConfigurationItem> clazz;

        public ClassBasedTypeDefinition(Class<? extends ConfigurationItem> ci) {
            this.type = type(ci);
            this.superType = type(ci.getSuperclass());
            this.interfaces = getAllInterfaces(ci).stream().filter(TypeDefinitions::isClassACi).map(Type::valueOf).collect(toList());
            this.clazz = ci;
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            return new ClassBasedLocalDescriptor(clazz);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitions.class);
}
