package com.xebialabs.xlplatform.synthetic.xml;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import static com.xebialabs.deployit.booter.local.utils.Closeables.closeQuietly;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI;

public class SyntheticXmlDocument {

    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = createDocumentBuilderFactory();
    private static final Logger logger = LoggerFactory.getLogger(SyntheticXmlDocument.class);

    private final List<TypeSpecification> types;
    private final List<TypeModificationSpecification> typeModifications;

    private SyntheticXmlDocument(List<TypeSpecification> types, List<TypeModificationSpecification> typeModifications) {
        this.types = types;
        this.typeModifications = typeModifications;
    }

    public static SyntheticXmlDocument read(URL syntheticXML) throws IOException {
        Element docElement = readSyntheticDocument(syntheticXML).getDocumentElement();
        return create(docElement);
    }

    public List<TypeSpecification> getTypes() {
        return types;
    }

    public List<TypeModificationSpecification> getTypeModifications() {
        return typeModifications;
    }

    //
    // Parse XML
    //

    private static DocumentBuilderFactory createDocumentBuilderFactory() {
        try {
            SchemaFactory schemaFactory = SchemaFactory.newInstance(W3C_XML_SCHEMA_NS_URI);
            Schema syntheticSchema = schemaFactory.newSchema(MODULE$.classLoader().getResource("synthetic.xsd"));
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
            documentBuilderFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
            documentBuilderFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");
            documentBuilderFactory.setNamespaceAware(true);
            documentBuilderFactory.setSchema(syntheticSchema);
            return documentBuilderFactory;
        } catch (SAXException exc) {
            throw new IllegalStateException("Cannot read schema synthetic.xsd", exc);
        }
    }

    private static Document readSyntheticDocument(final URL syntheticXML) throws IOException {
        InputStream syntheticXMLStream = syntheticXML.openStream();
        try {
            final boolean[] validationErrorsFound = new boolean[1];
            DocumentBuilder builder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            builder.setErrorHandler(new ErrorHandler() {
                @Override
                public void warning(SAXParseException exc) {
                    logger.warn("Warning while parsing " + syntheticXML, exc);
                }

                @Override
                public void error(SAXParseException exc) {
                    logger.error("Error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }

                @Override
                public void fatalError(SAXParseException exc) {
                    logger.error("Fatal error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }
            });
            Document doc = builder.parse(syntheticXMLStream);
            if (validationErrorsFound[0]) {
                throw new IllegalArgumentException("One or more errors were found while parsing " + syntheticXML);
            }

            return doc;
        } catch (RuntimeException | ParserConfigurationException | SAXException exc) {
            throw new IllegalStateException("Cannot read synthetic configuration " + syntheticXML, exc);
        } finally {
            closeQuietly(syntheticXMLStream);
        }
    }

    private static SyntheticXmlDocument create(Element docElement) {

        List<TypeSpecification> types = new ArrayList<>();
        List<TypeModificationSpecification> typeModifications = new ArrayList<>();

        XmlUtils.childByName(docElement, "type"::equals)
                .forEachRemaining(element -> types.add(new XmlTypeSpecification(element)));
        XmlUtils.childByName(docElement, "type-modification"::equals)
                .forEachRemaining(element -> typeModifications.add(new XmlTypeModificationSpecification(element)));

        return new SyntheticXmlDocument(types, typeModifications);
    }
}
