package com.xebialabs.deployit.booter.local;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.overthere.RuntimeIOException;

import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static java.lang.String.format;

public class PluginVersions {

    public static final String TYPE_SQL = "sql";

    private static class PluginVersion {
        final String type;
        final String version;

        private PluginVersion(final String type, final String version) {
            this.type = type;
            this.version = version;
        }
    }

    private static final Map<String, PluginVersion> PLUGIN_VERSIONS = new LinkedHashMap<>();

    static void init() {
        try {
            Enumeration<URL> resources = MODULE$.classLoader().getResources("plugin-version.properties");
            forEachRemaining(resources, PluginVersions::readPluginVersion);
        } catch (IOException | RuntimeIOException e) {
            logger.error("Couldn't read plugin-version.properties", e);
            throw new IllegalStateException(e);
        }
    }

    public static String getVersionFor(String plugin) {
        PluginVersion pluginVersion = getPluginVersionFor(plugin);
        if (pluginVersion == null) {
            return null;
        }
        return pluginVersion.version;
    }

    public static boolean hasTypeFor(String plugin, String type) {
        PluginVersion pluginVersion = getPluginVersionFor(plugin);
        if (pluginVersion == null || pluginVersion.type == null) {
            return TYPE_SQL.equals(type);
        }
        return type.equals(pluginVersion.type);
    }

    public static PluginVersion getPluginVersionFor(String plugin) {
        return PLUGIN_VERSIONS.get(plugin);
    }

    public static Set<String> getRegisteredPlugins() {
        return PLUGIN_VERSIONS.keySet();
    }

    private static void readPluginVersion(URL url) {
        try (InputStream input = url.openStream()) {
            Properties properties = new Properties();
            properties.load(input);
            String plugin = properties.getProperty("plugin");
            String type = properties.getProperty("type");
            String version = properties.getProperty("version");
            if (plugin == null) {
                throw new IllegalStateException(format("Couldn't read plugin-version.properties: plugin property is missing in '%s'.", url));
            }
            PLUGIN_VERSIONS.put(plugin, new PluginVersion(type, version));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private static <T> void forEachRemaining(Enumeration<T> e, Consumer<? super T> c) {
        while(e.hasMoreElements()) c.accept(e.nextElement());
    }

    private static final Logger logger = LoggerFactory.getLogger(PluginVersions.class);
}
