package com.xebialabs.deployit.booter.local;

import java.io.File;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;

final class GlobalContext {

    private static final Map<String, Value> context = new ConcurrentHashMap<>();
    private static final AtomicReference<GlobalContextManager> globalContextManagerReference = new AtomicReference<>(
            new LocalGlobalContextManager().withContext(context)
    );

    public static final File DEFAULTS = new File("conf", "deployit-defaults.properties");

    private GlobalContext() {
    }

    static void setManager(GlobalContextManager globalContextManager) {
        globalContextManagerReference.set(
                globalContextManager.withContext(context)
        );
    }

    static GlobalContextManager getManager() {
        return globalContextManagerReference.get();
    }

    static void register(PropertyDescriptor pd, String defaultValue) {
        if (defaultValue == null && !pd.isHidden()) return;
        context.put(pd.getFqn(), new Value(pd.getDescription(), defaultValue));
    }

    public static void register(PropertyDescriptor pd, PropertyDescriptor inheritedFrom) {
        context.put(pd.getFqn(), new InheritedValue(pd.getDescription(), inheritedFrom.getFqn(), pd.isHidden()));
    }

    static String lookup(PropertyDescriptor pd) {
        return lookup(pd.getFqn());
    }

    private static String lookup(String key) {
        Value value = context.get(key);
        if (value == null) return null;
        return value.getValue();
    }

    static class InheritedValue extends Value {
        private final String superTypeProperty;
        private final boolean shouldWrite;

        InheritedValue(String description, String superTypeProperty, boolean shouldWrite) {
            super(description, "");
            this.superTypeProperty = superTypeProperty;
            this.shouldWrite = shouldWrite;
        }

        @Override
        public String getValue() {
            return lookup(superTypeProperty);
        }

        @Override
        boolean isShouldWrite() {
            return shouldWrite;
        }

        String getSuperTypeProperty() {
            return superTypeProperty;
        }
    }

    static class Value {
        Value(String description, String value) {
            this.description = description;
            this.value = value;
        }

        final String description;
        final String value;

        boolean isExplicit() {
            return false;
        }

        boolean isShouldWrite() {
            return true;
        }

        public String getValue() {
            return value;
        }
    }

    static class ExplicitValue extends Value {
        ExplicitValue(String description, String value) {
            super(description, value);
        }

        @Override
        boolean isExplicit() {
            return true;
        }
    }
}
