package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

import static com.xebialabs.deployit.booter.local.utils.Closeables.closeQuietly;

class GlobalContext {
    private static final Logger logger = LoggerFactory.getLogger(GlobalContext.class);

    private static final Function<String, Value> COMPUTE_FUNCTION = s -> new Value("", null);

    private static final Map<String, Value> context = new ConcurrentHashMap<>();

    public static final File DEFAULTS = new File("conf", "deployit-defaults.properties");

    private GlobalContext() {
    }

    static void register(PropertyDescriptor pd, String defaultValue) {
        if (defaultValue == null && !pd.isHidden()) return;
        context.put(pd.getFqn(), new Value(pd.getDescription(), defaultValue));
    }

    public static void register(PropertyDescriptor pd, PropertyDescriptor inheritedFrom) {
        context.put(pd.getFqn(), new InheritedValue(pd.getDescription(), inheritedFrom.getFqn(), pd.isHidden()));
    }


    static String lookup(PropertyDescriptor pd) {
        return lookup(pd.getFqn());
    }

    private static String lookup(String key) {
        Value value = context.get(key);
        if (value == null) return null;
        return value.getValue();
    }

    static void loadStoredDefaults(File configFile) {
        if (!configFile.exists()) {
            logger.info("Could not find '" + configFile + "', continuing without loading defaults");
            return;
        } else if (configFile.isDirectory()) {
            logger.error(configFile + " is a directory but should be a plain file, aborting!");
            throw new IllegalStateException("Please remove " + configFile + " and try again.");
        }
        try (BufferedReader br = new BufferedReader(new FileReader(configFile))){
            br.lines().forEach(GlobalContext::processLine);
        } catch (IOException e) {
            throw new IllegalArgumentException("Could not read '" + configFile.toString() + "'", e);
        }
    }

    static void loadStoredDefaults(Properties properties) {
        if (properties.isEmpty()) {
            logger.info("Properties were empty', continuing without loading defaults");
            return;
        }
        processProperties(properties);
    }

    private static void processLine(String line) {
        if (!line.startsWith("#") && line.contains("=")) {
            int i = line.indexOf("=");
            String key = line.substring(0, i);
            Value value = context.computeIfAbsent(key, COMPUTE_FUNCTION);
            context.put(key, new ExplicitValue(value.description, line.substring(i + 1)));
        }
    }

    private static void processProperties(Properties properties) {
        properties.forEach((key, value) -> {
            Value contextValue = context.computeIfAbsent(key.toString(), COMPUTE_FUNCTION);
            context.put(key.toString(), new ExplicitValue(contextValue.description, value.toString()));
        });
    }


    public static void storeDefaults() {
        storeDefaults(DEFAULTS);
    }

    static void storeDefaults(File configFile) {
        if (configFile != null) {
            BufferedWriter bufferedWriter = null;
            try {
                if (!configFile.getParentFile().exists() || !configFile.getParentFile().isDirectory()) {
                    logger.warn("Not writing {} because the directory does not exist", configFile);
                    return;
                }
                bufferedWriter = new BufferedWriter(new FileWriter(configFile));
                bufferedWriter.append("# Note: If you modify this file, you must restart the XL Deploy server.");
                bufferedWriter.newLine();
                bufferedWriter.newLine();
                List<String> keys = new ArrayList<>(context.keySet());
                Collections.sort(keys);
                for (String k : keys) {
                    Value v = context.computeIfAbsent(k,COMPUTE_FUNCTION);
                    if (v.isShouldWrite()) {
                        if (v.description != null && !v.description.isEmpty()) {
                            bufferedWriter.append("# ").append(v.description);
                            if (v instanceof InheritedValue) {
                                InheritedValue inheritedValue = (InheritedValue) v;
                                bufferedWriter.append(" (inherited from: ").append(inheritedValue.superTypeProperty).append(")");
                            }
                            bufferedWriter.newLine();
                        }
                        bufferedWriter.append(v.isExplicit() ? "" : "#").append(k).append('=').append(v.getValue());
                        bufferedWriter.newLine();
                    }
                }
            } catch (FileNotFoundException e) {
                logger.error("Could not start writing to '" + configFile + "'", e);
            } catch (IOException e) {
                logger.error("Could not write to '" + configFile + "'", e);
            } finally {
                closeQuietly(bufferedWriter);
            }
        }
    }

    static void validateValues() {
        validateValues(DEFAULTS);
    }

    static void validateValues(File configFile) {
        boolean valid = true;
        List<String> messages = new ArrayList<>();
        for (String prop : context.keySet()) {
            int i = prop.lastIndexOf('.');
            String type = prop.substring(0, i);
            String propertyName = prop.substring(i + 1);

            Type tType = Type.valueOf(type);
            if (LocalDescriptorRegistry.exists(tType)) {
                Descriptor descriptor = DescriptorRegistry.getDescriptor(tType);
                if (descriptor.isVirtual()) continue;
                PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(propertyName);
                if (propertyDescriptor != null) {
                    try {
                        propertyDescriptor.getDefaultValue();
                        if (propertyDescriptor.isRequired() && propertyDescriptor.isHidden() && Strings.isEmpty(lookup(propertyDescriptor))) {
                            valid = false;
                            messages.add(String.format("Cannot register empty default value for hidden required property [%s]", prop));
                        }
                    } catch (RuntimeException re) {
                        valid = false;
                        messages.add(String.format("Incorrect default registered: [%s] with value %s", prop, context.computeIfAbsent(prop, COMPUTE_FUNCTION).getValue()));
                    }
                }
            }
        }

        if (!valid) {
            String join = messages.stream().reduce((s, t) -> s + "\n" + t).get();
            logger.error(join);
            String errorMessage = "Could not initialize the default values, please look at the log and correct the configuration file " + "\n" + join;
            if (configFile != null) {
                errorMessage = "Could not initialize the default values, please look at the log and correct the defaults file: " + configFile + "\n" + join;
            }
            throw new IllegalStateException(errorMessage);
        }
    }

    static class InheritedValue extends Value {
        private final String superTypeProperty;
        private final boolean shouldWrite;

        InheritedValue(String description, String superTypeProperty, boolean shouldWrite) {
            super(description, "");
            this.superTypeProperty = superTypeProperty;
            this.shouldWrite = shouldWrite;
        }

        @Override
        public String getValue() {
            return lookup(superTypeProperty);
        }

        @Override
        boolean isShouldWrite() {
            return shouldWrite;
        }
    }

    static class Value {
        Value(String description, String value) {
            this.description = description;
            this.value = value;
        }

        final String description;
        final String value;

        boolean isExplicit() {
            return false;
        }

        boolean isShouldWrite() {
            return true;
        }

        public String getValue() {
            return value;
        }
    }

    static class ExplicitValue extends Value {
        ExplicitValue(String description, String value) {
            super(description, value);
        }

        @Override
        boolean isExplicit() {
            return true;
        }
    }
}
