package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.generator.TypeGenerators;
import com.xebialabs.deployit.io.ConfigurationResource;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Class responsible for booting all the plugins. Responsible for: - Registering all the Descriptors - Setting up the
 * Global Context.
 */
public class LocalBooter {

    private static final AtomicBoolean isBooted = new AtomicBoolean(false);
    private static final AtomicBoolean initialized = new AtomicBoolean(false);

    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void boot() {
        try {
            boot(true, Collections.<String>emptyList(), false, null);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void boot(File propertiesFile) {
        try {
            boot(true, Collections.<String>emptyList(), false, propertiesFile);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }


    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void bootFromProperties(Properties properties, File propertiesFile) {
        try {
            if (!isBooted.getAndSet(true)) {
                TypeDefinitions typeDefinitions = bootStage1(Collections.<String>emptyList(), false);
                GlobalContextInitializer.init(properties, propertiesFile);
                bootStage2(typeDefinitions);
            }

            if (!initialized.get()) {
                throw new IllegalStateException("The DescriptorRegistry has been booted, but is not initialized. Please check the logs for any errors.");
            }
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System. Setting everything up. You can also specify extra packages to scan for classes
     * with XL Deploy annotations.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void boot(List<String> packagesToScan) {
        try {
            boot(true, packagesToScan, false, null);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. Useful for tests where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext() {
        boot(false, Collections.<String>emptyList(), false);
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. You can also specify extra packages to scan for classes with XL Deploy annotations. Useful for tests
     * where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext(List<String> packagesToScan) {
        boot(false, packagesToScan, false);
    }

    public static void rebootWithoutGlobalContext(List<String> packagesToScan) {
        unboot();
        boot(false, packagesToScan, true);
    }

    private synchronized static void unboot() {
        DescriptorRegistry.remove(LocalDescriptorRegistry.LOCAL_ID);
        DelegateRegistry.unboot();
        isBooted.set(false);
        initialized.set(false);
    }

    private synchronized static void boot(boolean withGlobalContext, List<String> packagesToScan, Boolean forceScan) {
        boot(withGlobalContext, packagesToScan, forceScan, null);
    }

    private synchronized static void boot(boolean withGlobalContext, List<String> packagesToScan, Boolean forceScan, File propertiesFile) {
        if (!isBooted.getAndSet(true)) {
            try {
                TypeDefinitions typeDefinitions = bootStage1(packagesToScan, forceScan);
                if (withGlobalContext) {
                    if (propertiesFile != null) {
                        GlobalContextInitializer.init(propertiesFile);
                    } else {
                        GlobalContextInitializer.init();
                    }
                }
                bootStage2(typeDefinitions);
            } catch (IllegalStateException e) {
                unboot(); // clear local booter state on failure
                throw e;
            }
        }

        if (!initialized.get()) {
            throw new IllegalStateException("The DescriptorRegistry has been booted, but is not initialized. Please check the logs for any errors.");
        }
    }

    private synchronized static TypeDefinitions bootStage1(List<String> packagesToScan, Boolean forceScan) {
        LocalDescriptorRegistry registry = new LocalDescriptorRegistry();
        DescriptorRegistry.add(registry);

        Configuration configuration = Configuration.config()
                .scan("com.xebialabs")
                .scan("ext.deployit") // Deployit Extensions
                .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner());

        for (String pkg : packagesToScan) {
            logger.debug("Scanning additional package '{}' for types, rules etc.", pkg);
            configuration.scan(pkg);
        }

        Scannit.boot(configuration, forceScan);
        DelegateRegistry.boot();
        TypeDefinitions typeDefinitions = new TypeDefinitions(registry);
        TypeSystemBootstrapper typeSystemBootstrapper = new TypeSystemBootstrapper(new TypeGenerators(Scannit.getInstance()));
        typeSystemBootstrapper.bootstrap(typeDefinitions);
        Verifications verifications = new Verifications();
        DelegateRegistry.verify(verifications);
        typeSystemBootstrapper.verifyTypes(verifications);
        verifications.done();
        return typeDefinitions;
    }

    private synchronized static void bootStage2(TypeDefinitions typeDefinitions) {
        PluginVersions.init();
        ConfigurationItemPostProcessors.registerPostProcessors();
        CiRoots.findRoots(typeDefinitions);
        initialized.set(true);
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalBooter.class);
}
