package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.util.*;

import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.getAllInterfaces;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredStringAttribute;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.getRequiredTypeAttribute;
import static com.xebialabs.overthere.util.OverthereUtils.checkNotNull;
import static com.xebialabs.overthere.util.OverthereUtils.checkState;
import static java.util.stream.Collectors.toList;

public class TypeDefinitions {
    private final Map<Type, TypeDefinition> typeDefs = new HashMap<>();
    private LocalDescriptorRegistry registry;

    public TypeDefinitions(LocalDescriptorRegistry registry) {
        this.registry = registry;
    }

    void defineType(Class clazz) {
        @SuppressWarnings("unchecked")
        ClassBasedTypeDefinition typeDef = new ClassBasedTypeDefinition(clazz);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    public void addTypeDef(TypeDefinition typeDef) {
        checkState(!typeDefs.containsKey(typeDef.type), "Trying to register duplicate definition for type [%s]. Existing: [%s], duplicate: [%s]", typeDef.type, typeDefs.get(typeDef.type), typeDef);
        typeDefs.put(typeDef.type, typeDef);
    }

    void defineType(Element element) {
        SyntheticBasedTypeDefinition typeDef = new SyntheticBasedTypeDefinition(element);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);

    }

    void defineGeneratedType(Element element, Element owner) {
        GeneratedTypeDefinition typeDef = new GeneratedTypeDefinition(owner, element);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    void defineGeneratedParameterType(final Element methodDef, final Element type) {
        GeneratedParameterTypeDefinition typeDef = new GeneratedParameterTypeDefinition(type, methodDef);
        addTypeDef(typeDef);
        logger.debug("Found: {}", typeDef);
    }

    void modifyType(Element element) {
        TypeMod typeMod = new TypeMod(getRequiredTypeAttribute(element, "type"), element);
        checkState(typeDefs.containsKey(typeMod.type), "Detected type-modification for non-existing type [%s]", typeMod.type);
        typeDefs.get(typeMod.type).typeModifications.push(typeMod);
        logger.debug("Found: {}", typeMod);
    }

    public Collection<TypeDefinition> getDefinitions() {
        return new ArrayList<>(typeDefs.values());
    }

    TypeDefinition getDefinition(Type type) {
        return checkNotNull(typeDefs.get(type), "Could not find a type definition associated with type [%s]", type);
    }

    private static boolean isClassACi(Class<?> clazz) {
        return clazz != null && ConfigurationItem.class.isAssignableFrom(clazz);
    }

    private static Type type(Class<?> clazz) {
        if (isClassACi(clazz)) {
            return Type.valueOf(clazz);
        } else {
            return null;
        }
    }

    public static Type generatedParameterType(final Type type, final String name) {
        return Type.valueOf(type.toString() + "_" + name);
    }

    class TypeMod {
        private Type type;
        private Element modification;

        protected TypeMod(final Type type, final Element modification) {
            this.type = type;
            this.modification = modification;
        }

        @Override
        public String toString() {
            return this.getClass().getSimpleName() + "[" + type + "]";
        }
    }

    public abstract class TypeDefinition {
        Type type;
        Type superType;
        Deque<TypeMod> typeModifications = new ArrayDeque<>();

        public void setType(Type type) {
            this.type = type;
        }

        public Type getType() {
            return type;
        }

        public Type getSuperType() {
            return superType;
        }

        public void setSuperType(Type superType) {
            this.superType = superType;
        }

        @Override
        public String toString() {
            return this.getClass().getSimpleName() + "[" + type + "]";
        }

        public final void register(TypeDefinitions typeDefinitions) {
            if (LocalDescriptorRegistry.exists(type)) {
                return;
            }

            if (superType != null && !LocalDescriptorRegistry.exists(superType)) {
                typeDefinitions.getDefinition(superType).register(typeDefinitions);
            }

            LocalDescriptor descriptor = createDescriptor(typeDefinitions);
            registry.register(descriptor);

            applyTypeModifications(descriptor);
        }

        protected abstract LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions);

        void applyTypeModifications(LocalDescriptor descriptor) {
            for (TypeMod typeModification : typeModifications) {
                descriptor.applyTypeModification(typeModification.modification);
            }
        }

        final void registerTypeTree() {
            registerAsSubtype(type);
        }

        protected void registerAsSubtype(Type type) {
            if (this.superType != null) {
                registerSuperType(superType, type);
                getDefinition(this.superType).registerAsSubtype(type);
            }
        }

        protected void registerSuperType(Type superType, Type type) {
            registry.registerSubtype(superType, type);
        }
    }

    class ClassBasedTypeDefinition extends TypeDefinition {
        private Class<? extends ConfigurationItem> clazz;
        private List<Type> interfaces = new ArrayList<>();

        public ClassBasedTypeDefinition(Class<? extends ConfigurationItem> ci) {
            this.type = type(ci);
            this.superType = type(ci.getSuperclass());
            this.interfaces = getAllInterfaces(ci).stream().filter(TypeDefinitions::isClassACi).map(Type::valueOf).collect(toList());
            this.clazz = ci;
        }


        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            for (Type anInterface : interfaces) {
                if (!LocalDescriptorRegistry.exists(anInterface)) {
                    typeDefinitions.getDefinition(anInterface).register(typeDefinitions);
                }
            }

            return new ClassBasedLocalDescriptor(clazz);
        }

        @Override
        protected void registerAsSubtype(Type type) {
            super.registerAsSubtype(type);
            for (Type anInterface : interfaces) {
                registerSuperType(anInterface, type);
            }
        }
    }

    class SyntheticBasedTypeDefinition extends TypeDefinition {
        private Element element;

        public SyntheticBasedTypeDefinition(Element element) {
            this.element = element;
            this.type = getRequiredTypeAttribute(element, "type");
            this.superType = getRequiredTypeAttribute(element, "extends");
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            return new SyntheticLocalDescriptor(element);
        }
    }

    class GeneratedTypeDefinition extends SyntheticBasedTypeDefinition {
        Type owner;

        private GeneratedTypeDefinition(Element owner, Element generatedElement) {
            super(generatedElement);
            this.owner = getRequiredTypeAttribute(owner, "type");
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            if (!LocalDescriptorRegistry.exists(owner)) {
                typeDefinitions.getDefinition(owner).register(typeDefinitions);
            }

            return new GenerateDeployableLocalDescriptor((SyntheticLocalDescriptor) LocalDescriptorRegistry.getDescriptor(owner));
        }
    }

    class GeneratedParameterTypeDefinition extends TypeDefinition {

        private final Type owner;
        private final Element methodDef;

        public GeneratedParameterTypeDefinition(final Element owner, final Element methodDef) {
            this.owner = getRequiredTypeAttribute(owner, "type");
            this.methodDef = methodDef;
            String methodName = getRequiredStringAttribute(methodDef, "name");
            this.type = generatedParameterType(this.owner, methodName);
            this.superType = Type.valueOf(Parameters.class);
        }

        @Override
        protected LocalDescriptor createDescriptor(final TypeDefinitions typeDefinitions) {
            if (!LocalDescriptorRegistry.exists(owner)) {
                typeDefinitions.getDefinition(owner).register(typeDefinitions);
            }

            return new GeneratedParameterLocalDescriptor(this.type, methodDef);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeDefinitions.class);
}
