package com.xebialabs.deployit.booter.local;

import java.util.*;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.xebialabs.overthere.util.OverthereUtils.checkArgument;
import static com.xebialabs.overthere.util.OverthereUtils.checkState;


public class LocalDescriptorRegistry extends DescriptorRegistry {

    private static Logger logger = LoggerFactory.getLogger(LocalDescriptorRegistry.class);

    public static final DescriptorRegistryId LOCAL_ID = new LocalDescriptorRegistryId();

    private final Map<Type, Descriptor> descriptors = new HashMap<>();

    private final SealedDescriptorHierarchy subtypes = new SealedDescriptorHierarchy();

    protected LocalDescriptorRegistry() {
        super(LOCAL_ID);
    }

    void register(Descriptor descriptor) {
        checkState(!this.descriptors.containsKey(descriptor.getType()), "The type [%s] is already registered in XL Deploy.", descriptor.getType());
        logger.debug("Registered ConfigurationItem: {}", descriptor);

        this.descriptors.put(descriptor.getType(), descriptor);
    }

    void registerSubtype(Type supertype, Type subtype) {
        checkState(!supertype.equals(subtype), "Cannot register [%s] as its own subtype.", supertype);
        this.subtypes.registerSubType(supertype, subtype);
    }

    @Override
    protected boolean isLocal() {
        return true;
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        return Collections.unmodifiableCollection(descriptors.values());
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        return Collections.unmodifiableCollection(subtypes.getSubtypes(supertype));
    }

    @Override
    protected Descriptor _getDescriptor(Type type) {
        checkState(!descriptors.isEmpty(), "DescriptorRegistry not booted");
        checkArgument(exists(type), "Unknown type [%s]", type);
        return descriptors.get(type);
    }

    @Override
    protected boolean _exists(Type type) {
        return descriptors.containsKey(type);
    }

    private static class SealedDescriptorHierarchy {
        private Map<Type, Set<Type>> hierarchy = new HashMap<>();
        private ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

        void registerSubType(Type supertype, Type subtype) {
            lock.writeLock().lock();
            try {
                if (!hierarchy.containsKey(supertype)) {
                    hierarchy.put(supertype, new LinkedHashSet<>());
                }
                hierarchy.get(supertype).add(subtype);
            } finally {
                lock.writeLock().unlock();
            }
        }

        Collection<Type> getSubtypes(Type supertype) {
            lock.readLock().lock();
            try {
                Set<Type> types = hierarchy.get(supertype);
                if (types != null) {
                    return new ArrayList<>(types);
                } else {
                    return Collections.emptyList();
                }
            } finally {
                lock.readLock().unlock();
            }
        }
    }
}

