/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.xstream;

import java.util.Map;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.engine.api.execution.*;

import static com.xebialabs.xltype.serialization.xstream.Converters.writeNode;

public abstract class AbstractTaskWithBlockConverter implements Converter {

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        TaskWithBlock task = (TaskWithBlock) source;
        writer.addAttribute("id", task.getId());
        writer.addAttribute("failures", Integer.toString(task.getFailureCount()));
        if (task.getState() != null) { // cater for com.xebialabs.deployit.engine.tasker.repository.ActiveTasks
            writer.addAttribute("state", task.getState().name());
        }
        writer.addAttribute("owner", task.getOwner());
        if (task.getWorkerId() != null) {
            writer.addAttribute("workerId", task.getWorkerId().toString());
        }

        if (task.getBlock() != null) {
            writer.addAttribute("type", FetchMode.FULL.name());
        } else {
            writer.addAttribute("type", FetchMode.SUMMARY.name());
        }

        writeNode("description", task.getDescription(), writer);

        if (task.getStartDate() != null) {
            writeNode("startDate", new DateTimeAdapter().marshal(task.getStartDate()), writer);
        }
        if (task.getCompletionDate() != null) {
            writeNode("completionDate", new DateTimeAdapter().marshal(task.getCompletionDate()), writer);
        }
        if (task.getScheduledDate() != null) {
            writeNode("scheduledDate", new DateTimeAdapter().marshal(task.getScheduledDate()), writer);
        }

        writer.startNode("activeBlocks");
        if (task.getActiveBlocks() != null) {
            for (String current : task.getActiveBlocks()) {
                if (current != null) {
                    writeNode("active", current, writer);
                }
            }
        }
        writer.endNode();

        Map<String, String> metadata = task.getMetadata();
        writer.startNode("metadata");
        for (Map.Entry<String, String> e : metadata.entrySet()) {
            writeNode(e.getKey(), e.getValue(), writer);
        }
        writer.endNode();

        if (task.getBlock() != null) {
            writePhaseContainer(writer, context, task.getBlock());
        }

        if (task.getPackageDependencies() != null) {
            writeTaskPackageDependencies(writer, task);
        }
    }

    protected void writePhaseContainer(HierarchicalStreamWriter writer, MarshallingContext context, BlockState block) {
        writer.startNode("block");
        Converters.writePhaseContainer((PhaseContainerState) block, writer, context, false);
        writer.endNode();
    }

    protected void writeTaskPackageDependencies(HierarchicalStreamWriter writer, SerializableTask task) {
        writer.startNode("dependencies");
        for (TaskPackageDependency dependency : task.getPackageDependencies()) {
            writer.startNode("dependency");
            writeNode("application", dependency.getApplication(), writer);
            writeNode("version", dependency.getVersion(), writer);
            writer.endNode();
        }
        writer.endNode();
    }

    @Override
    public boolean canConvert(final Class type) {
        return TaskWithBlock.class.isAssignableFrom(type);
    }
}
