/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization;

import java.util.*;

import javax.xml.bind.DatatypeConverter;

import com.xebialabs.deployit.engine.api.dto.*;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;

public class ConfigurationItemConverter {

    private boolean readValidationMessages = false;
    private boolean writeValidationMessages = false;

    private final List<CiReference> references = new ArrayList<>();
    private final Map<String, ConfigurationItem> readCIs = new LinkedHashMap<>();

    public void setReadValidationMessages(boolean enabled) {
        this.readValidationMessages = enabled;
    }

    public void setWriteValidationMessages(boolean enabled) {
        this.writeValidationMessages = enabled;
    }

    public List<CiReference> getReferences() {
        return references;
    }

    public Map<String, ConfigurationItem> getReadCIs() {
        return readCIs;
    }

    //
    // Write CIs
    //

    public void writeCis(Collection<ConfigurationItem> cis, CiWriter writer) {
        writeCis(cis, writer, 0);
    }

    public void writeCis(Collection<ConfigurationItem> cis, CiWriter writer, int ciRefsFromLevel) {
        writer.startList();
        for (ConfigurationItem ci : cis) {
            writeCi(ci, writer, ciRefsFromLevel);
        }
        writer.endList();
    }

    public void writeCi(ConfigurationItem ci, CiWriter writer) {
        writeCi(ci, writer, 0);
    }

    public void writeCi(ConfigurationItem ci, CiWriter writer, int ciRefsFromLevel) {
        ci = unwrapValidated(ci);

        writer.startCi(ci.getType().toString(), ci.getId());
        if (ci instanceof BaseConfigurationItem) {
            String token = ((BaseConfigurationItem) ci).get$token();
            if (token != null && !token.trim().isEmpty()) {
                writer.token(token);
            }
            writer.ciAttributes(((BaseConfigurationItem) ci).get$ciAttributes());

        }
        writeValidationMessages(ci, writer);
        writeProperties(ci, writer, ciRefsFromLevel);
        writer.endCi();
    }

    @SuppressWarnings("deprecation")
    private ConfigurationItem unwrapValidated(ConfigurationItem ci) {
        if (ci instanceof ValidatedConfigurationItem) {
            return ((ValidatedConfigurationItem) ci).getWrapped();
        }
        return ci;
    }

    protected void writeProperties(ConfigurationItem ci, CiWriter writer, int ciRefsFromLevel) {
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (!propertyDescriptor.isHidden()) {
                writeProperty(ci, propertyDescriptor, writer, ciRefsFromLevel);
            }
        }
    }

    protected void writeProperty(ConfigurationItem ci, PropertyDescriptor property, CiWriter writer, int ciRefsFromLevel) {
        Object value = property.get(ci);
        if (value == null) return;
        writer.startProperty(property.getName());
        switch (property.getKind()) {
            case STRING:
                writeStringProperty(value, property, writer);
                break;
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                writer.valueAsString(value);
                break;
            case DATE:
                Calendar cal = GregorianCalendar.getInstance();
                cal.setTime((Date) value);
                writer.valueAsString(DatatypeConverter.printDateTime(cal));
                break;
            case CI:
                if (ciRefsFromLevel >= 1 && isContainedChild(value, ci)) {
                    writeCi((ConfigurationItem) value, writer);
                } else {
                    writeCiProperty(value, property, writer);
                }
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                writeCollectionOfStringProperty(value, property, writer);
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                if (ciRefsFromLevel >= 1 && property.isAsContainment()) {
                    @SuppressWarnings("unchecked")
                    Collection<ConfigurationItem> collection = (Collection<ConfigurationItem>) value;
                    writeCis(collection, writer, ciRefsFromLevel - 1);
                } else {
                    writeCollectionOfCiProperty(value, property, writer);
                }
                break;
            case MAP_STRING_STRING:
                writeMapStringStringProperty(value, property, writer);
                break;
        }
        writer.endProperty();
    }

    private static boolean isContainedChild(Object object, ConfigurationItem parent) {
        if (object instanceof ConfigurationItem) {
            ConfigurationItem ci = (ConfigurationItem) object;
            for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
                if (property.getKind() == PropertyKind.CI && property.isAsContainment() && parent.equals(property.get(ci))) {
                    return true;
                }
            }
        }

        return false;
    }

    protected void writeStringProperty(Object value, @SuppressWarnings("unused") PropertyDescriptor propertyDescriptor, CiWriter writer) {
        writer.valueAsString(value);
    }

    @SuppressWarnings("unchecked")
    protected void writeMapStringStringProperty(Object value, @SuppressWarnings("unused") PropertyDescriptor propertyDescriptor, CiWriter writer) {
        writer.mapAsStrings((Map<String, String>) value);
    }

    @SuppressWarnings("unchecked")
    protected void writeCollectionOfStringProperty(Object value, @SuppressWarnings("unused") PropertyDescriptor propertyDescriptor, CiWriter writer) {
        writer.valuesAsStrings((Collection<String>) value);
    }

    protected void writeCiProperty(Object value, @SuppressWarnings("unused") PropertyDescriptor propertyDescriptor, CiWriter writer) {
        writer.ciReference(getIdOfCi(value));
    }

    protected void writeCollectionOfCiProperty(Object value, @SuppressWarnings("unused") PropertyDescriptor propertyDescriptor, CiWriter writer) {
        Collection<?> cis = (Collection<?>) value;
        Collection<String> ids = new ArrayList<>(cis.size());
        for (Object ci : cis) {
            ids.add(getIdOfCi(ci));
        }
        writer.ciReferences(ids);
    }

    protected void writeValidationMessages(ConfigurationItem ci, CiWriter writer) {
        if (!writeValidationMessages || ci.get$validationMessages().isEmpty()) {
            return;
        }
        writer.validationMessages(ci.get$validationMessages());
    }

    //
    // Read CIs
    //

    public ConfigurationItem readCi(CiReader reader) {

        // Create a new CI
        Type type = type(reader.getType());
        String id = reader.getId();
        Descriptor descriptor = type.getDescriptor();
        if (descriptor == null) {
            throw new IllegalStateException("Encountered unknown CI type [" + type + "] for ConfigurationItem [" + id + "]");
        }
        ConfigurationItem ci = descriptor.newInstance(id);
        if (ci instanceof BaseConfigurationItem) {
            final BaseConfigurationItem baseConfigurationItem = (BaseConfigurationItem) ci;
            baseConfigurationItem.set$token(reader.getToken());
            baseConfigurationItem.set$ciAttributes(reader.getCiAttributes());
        }

        // Fill properties
        ci = readProperties(reader, descriptor, ci);

        // Keep track of all CIs parsed by this converter
        // (will be used to resolve references when parsing a list of CIs)
        readCIs.put(ci.getId(), ci);

        return ci;
    }

    public Type type(String typeName) {
        return Type.valueOf(typeName);
    }

    public List<ConfigurationItem> readCis(CiListReader reader) {
        List<ConfigurationItem> cis = new ArrayList<>();

        while(reader.hasMoreChildren()) {
            reader.moveIntoChild();
            ConfigurationItem ci = readCi(reader.getCurrentCiReader());
            cis.add(ci);
            reader.moveOutOfChild();
        }

        return cis;
    }

    // We need this because of the streaming nature of XStream -- there is no peek to see what kind of list we are actually parsing
    private List<Object> readCisOrReferences(CiListReader reader) {
        List<Object> cis = new ArrayList<>();

        while(reader.hasMoreChildren()) {
            reader.moveIntoChild();
            CiReader currentCiReader = reader.getCurrentCiReader();
            if (currentCiReader.isCiReference()) {
                cis.add(currentCiReader.getCiReference());
            } else {
                ConfigurationItem ci = readCi(currentCiReader);
                cis.add(ci);
            }
            reader.moveOutOfChild();
        }
        return cis;
    }

    protected ConfigurationItem readProperties(CiReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        ConfigurationItem toBeReturned = configurationItem;

        while (reader.hasMoreProperties()) {
            reader.moveIntoProperty();
            if (reader.getCurrentPropertyName().equals("validation-messages")) {
                if (readValidationMessages) {
                    List<ValidationMessage> validationMessages = reader.getValidationMessages();
                    if (configurationItem instanceof BaseConfigurationItem) {
                        configurationItem.get$validationMessages().addAll(validationMessages);
                    }
                    @SuppressWarnings("deprecation")
                    ValidatedConfigurationItem validatedCi = new ValidatedConfigurationItem(configurationItem);
                    validatedCi.setValidations(validationMessages);
                    toBeReturned = validatedCi;
                }
            } else {
                readProperty(reader, descriptor, configurationItem);
            }
            reader.moveOutOfProperty();
        }

        return toBeReturned;
    }


    protected void readProperty(CiReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        String propName = reader.getCurrentPropertyName();
        PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(propName);
        if (propertyDescriptor == null) {
            throw new IllegalStateException("Encountered unknown ConfigurationItem property [" + descriptor.getType() + "." + propName + "] for ci [" + configurationItem.getId() + "]");
        }
        switch (propertyDescriptor.getKind()) {
            case STRING:
                readStringProperty(configurationItem, propertyDescriptor, reader);
                break;
            case BOOLEAN:
            case INTEGER:
            case ENUM:
                propertyDescriptor.set(configurationItem, reader.getStringValue());
                break;
            case DATE:
                String stringValue = reader.getStringValue();
                if (stringValue != null && !stringValue.trim().isEmpty()) {
                    propertyDescriptor.set(configurationItem, DatatypeConverter.parseDateTime(stringValue).getTime());
                }
                break;
            case CI:
                readCiProperty(configurationItem, propertyDescriptor, reader);
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                readCollectionOfStringProperty(configurationItem, propertyDescriptor, reader);
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                readCollectionOfCiProperty(configurationItem, propertyDescriptor, reader);
                break;
            case MAP_STRING_STRING:
                readMapStringStringProperty(configurationItem, propertyDescriptor, reader);
                break;
        }
    }

    protected void readStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        propertyDescriptor.set(configurationItem, reader.getStringValue());
    }

    protected void readMapStringStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        Map<String, String> map = reader.getStringMap();
        propertyDescriptor.set(configurationItem, map);
    }

    protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        Collection<String> strings = reader.getStringValues();
        if (propertyDescriptor.getKind() == SET_OF_STRING) {
            strings = new LinkedHashSet<>(strings);
        }
        propertyDescriptor.set(configurationItem, strings);
    }

    protected static String getIdOfCi(Object object) {
        if (object instanceof String) {
            return (String) object;
        } else if (object instanceof ConfigurationItem) {
            return ((ConfigurationItem) object).getId();
        }

        throw new IllegalArgumentException(String.format("Value should be of type String or Configuration Item, but is %s (%s)", object.getClass().getName(), object));
    }

    protected void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        if (reader.isCiReference()) {
            String id = reader.getCiReference();
            references.add(new CiReference(configurationItem, propertyDescriptor, id));
        } else {
            // Get the nested reader
            ConfigurationItem child = readCi(reader.moveIntoNestedProperty());
            propertyDescriptor.set(configurationItem, child);
            // Move up if you're in a streamed reader
            reader.moveOutOfProperty();
        }
    }

    // TODO check with subclass for remote booter. It overrides this method, but needs to be updated in order to accept CI trees.
    protected void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        List<?> items = readCisOrReferences(reader.getCurrentCiListReader());
        if (items.isEmpty()) {
            return;
        }

        if (items.get(0) instanceof String) {
            @SuppressWarnings("unchecked")
            List<String> strings = (List<String>) items;
            references.add(new CiReference(configurationItem, propertyDescriptor, strings));
        } else {
            if (propertyDescriptor.getKind() == SET_OF_CI) {
                propertyDescriptor.set(configurationItem, new LinkedHashSet<>(items));
            } else {
                propertyDescriptor.set(configurationItem, items);
            }
        }
    }

    public void resolveReferences(Repository repository) {
        for (CiReference reference : getReferences()) {

            // Resolve IDs to CIs
            List<ConfigurationItem> resolvedCIs = new ArrayList<>();
            for (String id : reference.getIds()) {
                ConfigurationItem alsoRead = getReadCIs().get(id);
                if (alsoRead != null) {
                    resolvedCIs.add(alsoRead);
                } else {
                    resolvedCIs.add(repository.read(id));
                }
            }

            // Set referred CIs on the parsed CI
            reference.set(resolvedCIs);
        }
    }

}
