/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

public abstract class StepJsonConverter {

    private final DateTimeAdapter timeAdapter = new DateTimeAdapter();

    public String toJson(StepState step) {
        JsonWriter writer = new JsonWriter();

        writeJson(step, writer);

        return writer.toString();
    }

    public JsonWriter writeJson(StepState step, JsonWriter writer) {
        writer.object();
        writer.key("state").value(step.getState());
        writer.key("failures").value(step.getFailureCount());
        writer.key("skippable").value(step.getSkippable());
        writer.key("description").value(step.getDescription());
        if (step.getStartDate() != null) {
            writer.key("startDate").value(timeAdapter.marshal(step.getStartDate()));
        }
        if (step.getCompletionDate() != null) {
            writer.key("completionDate").value(timeAdapter.marshal(step.getCompletionDate()));
        }
        if (step.getLog() != null) {
            writer.key("log").value(step.getLog());
        }
        writer.endObject();

        return writer;
    }

    public StepState toStep(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            StepExecutionState state = StepExecutionState.valueOf(json.getString("state"));
            int failureCount = json.getInt("failures");
            String description = json.getString("description");
            boolean skippable = json.getBoolean("skippable");
            DateTime startDate = null;
            if (json.has("startDate")) {
                startDate = timeAdapter.unmarshal(json.getString("startDate"));
            }

            DateTime completionDate = null;
            if (json.has("completionDate")) {
                completionDate = timeAdapter.unmarshal(json.getString("completionDate"));
            }

            String log = null;
            if (json.has("log")) {
                log = json.getString("log");
            }

            return createStepState(state, failureCount, description, startDate, completionDate, log, skippable);

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    protected abstract StepState createStepState(StepExecutionState state, int failureCount, String description, DateTime startDate, DateTime completionDate, String log, boolean skippable);
}
