/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import java.io.StringWriter;
import java.io.Writer;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

public class CiJsonWriter implements CiWriter {

    protected final Writer writer;
    protected final JsonWriter json;
    protected final DateTimeAdapter dateTimeAdapter = new DateTimeAdapter();

    public CiJsonWriter() {
        this(new StringWriter());
    }

    public CiJsonWriter(Writer writer) {
        this.writer = writer;
        this.json = new JsonWriter(writer);
    }

    public CiJsonWriter(JsonWriter json) {
        this.json = json;
        this.writer = json.getWriter();
    }

    public Writer getWriter() {
        return writer;
    }

    @Override
    public String toString() {
        return writer.toString();
    }

    @Override
    public void startList() {
        json.array();
    }

    @Override
    public void endList() {
        json.endArray();
    }

    @Override
    public void startCi(String type, String id) {
        json.object();
        json.key("id").value(id);
        json.key("type").value(type);
    }

    @Override
    public void endCi() {
        json.endObject();
    }

    @Override
    public void token(String token) {
        json.key("$token").value(token != null ? token : "");
    }


    @Override
    public void ciAttributes(CiAttributes ciAttributes) {
        addStringAttribute("$createdBy", ciAttributes.getCreatedBy());
        addDateAttribute("$createdAt", ciAttributes.getCreatedAt());
        addStringAttribute("$lastModifiedBy", ciAttributes.getLastModifiedBy());
        addDateAttribute("$lastModifiedAt", ciAttributes.getLastModifiedAt());
    }

    @Override
    public void ciFileAttribute(String file) {
        addStringAttribute("file", file);
    }

    private void addStringAttribute(String attrName, String attrValue) {
        if (attrValue != null) {
            json.key(attrName).value(attrValue);
        }
    }

    private void addDateAttribute(String attrName, DateTime attrValue) {
        if (attrValue != null) {
            json.key(attrName).value(dateTimeAdapter.marshal(attrValue));
        }
    }
    @Override
    public void startProperty(String name) {
        json.key(name);
    }

    @Override
    public void endProperty() {
    }

    @Override
    public void valueAsString(Object value) {
        json.value(value);
    }

    @Override
    public void valuesAsStrings(Collection<?> values) {
        json.array();
        for (Object item : values) {
            valueAsString(item);
        }
        json.endArray();
    }

    @Override
    public void mapAsStrings(Map<?, ?> map) {
        json.object();
        for (Object key : map.keySet()) {
            json.key(String.valueOf(key));
            valueAsString(map.get(key));
        }
        json.endObject();
    }

    @Override
    public void ciReference(String reference) {
        valueAsString(reference);
    }

    @Override
    public void ciReferences(Collection<String> references) {
        valuesAsStrings(references);
    }

    @Override
    public void typedCiReference(ConfigurationItemId ci) {
        json.object();
        json.key("ci").value(ci.getId());
        if (ci.getType() != null) {
            json.key("type").value(ci.getType());
        }
        json.endObject();
    }

    @Override
    public void typedCiReferences(Collection<ConfigurationItemId> references) {
        json.array();
        for (ConfigurationItemId item : references) {
            typedCiReference(item);
        }
        json.endArray();
    }

    @Override
    public void validationMessages(List<ValidationMessage> validations) {
        json.key("validation-messages");
        json.array();
        for (ValidationMessage validation : validations) {
            json.object();
            json.key("level").value(validation.getLevel().name());
            json.key("ci").value(validation.getCiId());

            if (validation.getPropertyName() != null) {
                json.key("property").value(validation.getPropertyName());
            }
            json.key("message").value(validation.getMessage());
            json.endObject();
        }
        json.endArray();
    }

}
