package com.xebialabs.xltype.serialization.json;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;
import com.google.common.base.Function;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.SerializationException;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.newArrayListWithCapacity;
import static com.google.common.collect.Maps.newLinkedHashMap;

public class CiJsonReader implements CiReader {

    private final JSONObject json;

    private Iterator<String> propertyIterator;
    private String currentPropertyName;

    public CiJsonReader(JSONObject json) {
        this.json = json;
        this.propertyIterator = properties(json).iterator();
    }

    public static CiJsonReader create(String jsonObject) {
        try {
            return new CiJsonReader(new JSONObject(jsonObject));
        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse the following as a JSON object:\n" + jsonObject, e);
        }
    }

    //
    // CiReader implementation
    //

    @Override
    public String getType() {
        try {
            return json.getString("type");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getId() {
        try {
            return json.getString("id");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getToken() {
        return getValueOrNull("$token", AS_STRING);
    }

    @Override
    public CiAttributes getCiAttributes() {
        final String createdBy = getValueOrNull("$createdBy", AS_STRING);
        final DateTime createdAt = getValueOrNull("$createdAt", AS_DATE_TIME);
        final String lastModifiedBy = getValueOrNull("$lastModifiedBy", AS_STRING);
        final DateTime lastModifiedAt = getValueOrNull("$lastModifiedAt", AS_DATE_TIME);
        return new CiAttributes(createdBy, createdAt, lastModifiedBy, lastModifiedAt);
    }

    @Override
    public boolean hasMoreProperties() {
        return propertyIterator.hasNext();
    }

    @Override
    public void moveIntoProperty() {
        currentPropertyName = propertyIterator.next();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        try {
            return new CiJsonReader(json.getJSONObject(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public void moveOutOfProperty() {
        // Noop
    }

    @Override
    public String getCurrentPropertyName() {
        return currentPropertyName;
    }

    @Override
    public String getStringValue() {
        try {
            return json.getString(currentPropertyName);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<String> getStringValues() {
        try {
            JSONArray strings = json.getJSONArray(currentPropertyName);
            return toList(strings);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public Map<String, String> getStringMap() {
        try {
            JSONObject map = json.getJSONObject(currentPropertyName);
            return toMap(map);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public boolean isCiReference() {
        return currentPropertyName != null && json.optJSONObject(currentPropertyName) == null;
    }

    @Override
    public String getCiReference() {
        return getStringValue();
    }

    @Override
    public List<String> getCiReferences() {
        return getStringValues();
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        try {
            return new CiListJsonReader(json.getJSONArray(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        List<ValidationMessage> messages = newArrayList();
        try {
            JSONArray array = json.getJSONArray("validation-messages");
            for (int i = 0; i < array.length(); i++) {
                messages.add(toMessage(array.getJSONObject(i)));
            }
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
        return messages;
    }

    //
    // Util
    //

    private static Collection<String> properties(JSONObject object) {
        List<String> properties = newArrayList();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            if(!isSpecialKey(key)) {
                properties.add(key);
            }
        }

        return properties;
    }

    private static boolean isSpecialKey(String key) {
        return "id".equals(key) || "type".equals(key) || (key != null && key.startsWith("$"));
    }

    private static List<String> toList(JSONArray array) throws JSONException {
        List<String> strings = newArrayListWithCapacity(array.length());
        for (int i = 0; i < array.length(); i++) {
            strings.add(array.getString(i));
        }

        return strings;
    }

    private static Map<String, String> toMap(JSONObject object) throws JSONException {
        Map<String, String> map = newLinkedHashMap();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            map.put(key, object.getString(key));
        }

        return map;
    }

    private static ValidationMessage toMessage(JSONObject jsonObject) throws JSONException {
        return new ValidationMessage(
                jsonObject.getString("ci"),
                jsonObject.getString("property"),
                jsonObject.getString("message"));
    }

    private <T> T getValueOrNull(String key, Function<String, T> map) {
        try {
            if (json.has(key)) {
                return map.apply(json.getString(key));
            }
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
        return null;
    }

    private static Function<String, String> AS_STRING = new Function<String, String>() {
        @Override
        public String apply(final String input) {
            return input;
        }
    };

    private static Function<String, DateTime> AS_DATE_TIME = new Function<String, DateTime>() {

        private final DateTimeAdapter dateTimeAdapter = new DateTimeAdapter();

        @Override
        public DateTime apply(final String input) {
            if(input != null) {
                return dateTimeAdapter.unmarshal(input);
            }
            return null;
        }
    };
}
