package com.xebialabs.xltype.serialization.xstream;

import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import org.joda.time.DateTime;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.io.xml.PrettyPrintWriter;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiWriter;

import static com.google.common.base.Strings.nullToEmpty;

public class CiXStreamWriter implements CiWriter {

    private final HierarchicalStreamWriter writer;
    private final boolean suppressOuterElement;
    private Writer stringWriter = null;
    private final DateTimeAdapter dateTimeAdapter = new DateTimeAdapter();

    // The <list> element is only wrapped around the outer list of CIs. For properties that are LIST_OF_CI or SET_OF_CI,
    // we don't want to have a spurious <list> element. So somehow we need to maintain state whether to render this
    // <list> element is for. This is where the 'suppressListElement' and the corresponding stack come in.
    private boolean suppressListElement = false;
    private Deque<Boolean> suppressListStack = new ArrayDeque<>();

    public CiXStreamWriter() {
        stringWriter = new StringWriter();
        writer = new PrettyPrintWriter(stringWriter);
        suppressOuterElement = false;
    }

    public CiXStreamWriter(HierarchicalStreamWriter writer) {
        this(writer, false);
    }

    public CiXStreamWriter(HierarchicalStreamWriter writer, boolean suppressOuterElement) {
        this.writer = writer;
        this.suppressOuterElement = suppressOuterElement;
    }

    @Override
    public String toString() {
        if (stringWriter != null) {
            return stringWriter.toString();
        }
        return writer.toString();
    }

    @Override
    public void startList() {
        if (suppressListElement) {
            return;
        }

        writer.startNode("list");
    }

    @Override
    public void endList() {
        if (suppressListElement) {
            return;
        }

        writer.endNode();
    }

    @Override
    public void startCi(String type, String id) {
        if (!suppressOuterElement) {
            writer.startNode(type);
        }
        writer.addAttribute("id", id);
    }

    @Override
    public void endCi() {
        if (suppressOuterElement) {
            return;
        }
        writer.endNode();
    }

    @Override
    public void token(String token) {
        writer.addAttribute("token", nullToEmpty(token));
    }

    @Override
    public void ciAttributes(CiAttributes ciAttributes) {
        addStringAttribute("created-by", ciAttributes.getCreatedBy());
        addDateAttribute("created-at", ciAttributes.getCreatedAt());
        addStringAttribute("last-modified-by", ciAttributes.getLastModifiedBy());
        addDateAttribute("last-modified-at", ciAttributes.getLastModifiedAt());
    }

    @Override
    public void ciFileAttribute(String file) {
        addStringAttribute("file", file);
    }

    private void addStringAttribute(String attrName, String attrValue) {
        if (attrValue != null) {
            writer.addAttribute(attrName, attrValue);
        }
    }

    private void addDateAttribute(String attrName, DateTime attrValue) {
        if (attrValue != null) {
            writer.addAttribute(attrName, dateTimeAdapter.marshal(attrValue));
        }
    }

    @Override
    public void startProperty(String name) {
        writer.startNode(name);
        suppressListStack.push(suppressListElement);
        suppressListElement = true;
    }

    @Override
    public void endProperty() {
        suppressListElement = suppressListStack.pop();
        writer.endNode();
    }

    @Override
    public void valueAsString(Object value) {
        writer.setValue(String.valueOf(value));
    }

    @Override
    public void valuesAsStrings(Collection<?> values) {
        for (Object value : values) {
            writer.startNode("value");
            valueAsString(value);
            writer.endNode();
        }
    }

    @Override
    public void mapAsStrings(Map<?, ?> map) {
        for (Map.Entry<?, ?> entry : map.entrySet()) {
            writer.startNode("entry");
            writer.addAttribute("key", entry.getKey().toString());
            valueAsString(entry.getValue());
            writer.endNode();
        }
    }

    @Override
    public void ciReference(String reference) {
        writer.addAttribute("ref", reference);
    }

    @Override
    public void ciReferences(Collection<String> references) {
        for (String reference : references) {
            writer.startNode("ci");
            ciReference(reference);
            writer.endNode();
        }
    }

    @Override
    public void typedCiReference(ConfigurationItemId ci) {
        writer.startNode("ci");
        writer.addAttribute("ref", ci.getId());
        if (ci.getType() != null) {
            writer.addAttribute("type", ci.getType().toString());
        }
        writer.endNode();
    }

    @Override
    public void typedCiReferences(Collection<ConfigurationItemId> references) {
        writer.startNode("list");
        for (ConfigurationItemId item : references) {
            typedCiReference(item);
        }
        writer.endNode();
    }

    @Override
    public void validationMessages(List<ValidationMessage> validations) {
        writer.startNode("validation-messages");
        for (ValidationMessage validation : validations) {
            writer.startNode("validation-message");
            writer.addAttribute("ci", validation.getCiId());
            if (validation.getPropertyName() != null) {
                writer.addAttribute("property", validation.getPropertyName());
            }
            writer.setValue(validation.getMessage());
            writer.endNode();
        }
        writer.endNode();
    }
}