/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

@SuppressWarnings("deprecation")
public abstract class TaskJsonConverter {

    private final DateTimeAdapter timeAdapter = new DateTimeAdapter();
    private final StepJsonConverter stepConverter;

    public TaskJsonConverter(StepJsonConverter stepConverter) {
        this.stepConverter = stepConverter;
    }

    public String toJson(TaskState task) {
        JsonWriter writer = new JsonWriter();

        writer.object();
        writer.key("id").value(task.getId());
        writer.key("currentStep").value(task.getCurrentStepNr());
        writer.key("totalSteps").value(task.getNrSteps());
        writer.key("state").value(task.getState());
        writer.key("failures").value(task.getFailureCount());
        writer.key("description").value(task.getDescription());
        if (task.getStartDate() != null) {
            writer.key("startDate").value(timeAdapter.marshal(task.getStartDate()));
        }
        if (task.getCompletionDate() != null) {
            writer.key("completionDate").value(timeAdapter.marshal(task.getCompletionDate()));
        }
        if (task.getScheduledDate() != null) {
            writer.key("scheduledDate").value(timeAdapter.marshal(task.getScheduledDate()));
        }
        writer.key("metadata").object();
        for (Map.Entry<String, String> entry : task.getMetadata().entrySet()) {
            writer.key(entry.getKey()).value(entry.getValue());
        }
        writer.endObject();
        if (task instanceof TaskWithSteps) {
            writeSteps(writer, (TaskWithSteps) task);
        }
        writer.key("workerId").value(task.getWorkerId());

        writer.endObject();

        return writer.toString();
    }

    private void writeSteps(JsonWriter writer, TaskWithSteps task) {
        writer.key("steps");
        writer.array();
        for (StepState step : task.getSteps()) {
            stepConverter.writeJson(step, writer);
        }
        writer.endArray();
    }

    public TaskState toTask(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            String id = json.getString("id");
            int currentStepNr = json.getInt("currentStep");
            int nrSteps = json.getInt("totalSteps");
            TaskExecutionState state = TaskExecutionState.valueOf(json.getString("state"));
            int failureCount = json.getInt("failures");
            String description = json.getString("description");

            DateTime startDate = null;
            if (json.has("startDate")) {
                startDate = timeAdapter.unmarshal(json.getString("startDate"));
            }

            DateTime completionDate = null;
            if (json.has("completionDate")) {
                completionDate = timeAdapter.unmarshal(json.getString("completionDate"));
            }

            DateTime scheduledDate = null;
            if (json.has("scheduledDate")) {
                scheduledDate = timeAdapter.unmarshal(json.getString("completionDate"));
            }

            Map<String, String> metadata = new LinkedHashMap<>();
            JSONObject metadataJson = json.getJSONObject("metadata");
            Iterator<?> iter = metadataJson.keys();
            while (iter.hasNext()) {
                String key = iter.next().toString();
                metadata.put(key, metadataJson.getString(key));
            }

            return createTask(id, currentStepNr, nrSteps, state, failureCount, description, startDate, completionDate, scheduledDate, metadata);

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    protected abstract TaskState createTask(String id, int currentStepNr, int nrSteps, TaskExecutionState state, int failureCount, String description,
        DateTime startDate, DateTime completionDate, DateTime scheduledDate, Map<String, String> metadata);

}
