package com.xebialabs.deployit.documentation;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;

import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

import static com.google.common.base.Predicates.and;
import static com.google.common.base.Predicates.not;
import static com.google.common.base.Predicates.or;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Lists.newArrayList;

/**
 * Generates documentation for CIs found on the classpath.
 */
public class CiReferenceGenerator {

    static {
        LocalBooter.bootWithoutGlobalContext();
    }

    private List<String> prefixes;
    private List<Descriptor> cis;
    private CiReferenceHtmlWriter doc;

    public CiReferenceGenerator(List<String> prefixes, CiReferenceHtmlWriter writer) {
        this.prefixes = prefixes;
        this.cis = findDescriptors();
        this.doc = writer;
        this.doc.setCis(cis);
    }

    public void generateCiReference() {
        logger.info("Generating documentation for " + (!prefixes.isEmpty() ? "CIs with prefix '" + prefixes + "'" : "all CIs"));

        doc.startCiReference();

        ciSummary();
        ciDetails();

        doc.endCiReference();
    }

    public void generateDocument() {
        doc.startDocument();

        generateCiReference();

        doc.endDocument();
    }

    private void ciSummary() {
        doc.tocHeader();

        tableOfContents(filter(cis, CiPredicates.DEPLOYABLE), "Deployables");
        tableOfContents(filter(cis, CiPredicates.DEPLOYED), "Deployeds");
        tableOfContents(filter(cis, CiPredicates.CONTAINER), "Containers");
        tableOfContents(filter(cis, CiPredicates.otherCIs()), "Other Configuration Items");
    }

    private void ciDetails() {
        doc.detailsHeader();

        for (Descriptor ci : cis) {
            doc.ciDetailTitle(ci.getType());
            doc.ciTypeInfo(ci);
            doc.ciDescription(ci.getDescription());

            Collection<PropertyDescriptor> properties = sortProperties(ci.getPropertyDescriptors());
            doc.ciProperties(filter(properties, PropertyPredicates.PARENT), "Parent");
            doc.ciProperties(filter(properties, PropertyPredicates.CHILDREN), "Children");
            doc.ciProperties(filter(properties, PropertyPredicates.PUBLIC), "Public Properties");
            doc.ciProperties(filter(properties, PropertyPredicates.HIDDEN), "Hidden Properties");

            Collection<MethodDescriptor> methods = sortMethods(ci.getControlTasks());
            controlTasks(methods, "Control Tasks");

            doc.detailsSeparator();
        }
    }

    private void controlTasks(Collection<MethodDescriptor> methods, String category) {
        if (methods.isEmpty()) {
            return;
        }

        doc.startControlTask(category);

        int row = 0;
        for (MethodDescriptor method : methods) {
            doc.controlTaskMethod(method, row);
            row++;
        }

        doc.endControlTask();
    }

    private void tableOfContents(Collection<Descriptor> cis, String category) {
        if (cis.isEmpty()) {
            return;
        }

        doc.categoryHeader(category);

        doc.startToc();
        int row = 0;
        for (Descriptor ci : cis) {
            doc.tocEntry(ci, extractFirstSentence(ci.getDescription()), row);
            row++;
        }
        doc.endToc();
    }

    private static String extractFirstSentence(String text) {
        if (text == null) {
            return "";
        }
        if (text.indexOf(".") > -1) {
            return text.substring(0, text.indexOf("."));
        }
        return text;
    }

    private List<Descriptor> findDescriptors() {
        Collection<Descriptor> filtered = filter(DescriptorRegistry.getDescriptors(), new Predicate<Descriptor>() {
            @Override
            public boolean apply(Descriptor input) {
                return input.getDescription() != null && shouldInclude(input.getType());
            }
        });

        List<Descriptor> needsRefDoc = newArrayList(filtered);

        Collections.sort(needsRefDoc, new Comparator<Descriptor>() {
            @Override
            public int compare(Descriptor o1, Descriptor o2) {
                return o1.getType().toString().compareTo(o2.getType().toString());
            }
        });

        logger.info("Found {} CI descriptors to generate documentation for.", needsRefDoc.size());

        return needsRefDoc;
    }

    private boolean shouldInclude(Type type) {
        return prefixes.isEmpty() || prefixes.contains(type.getPrefix());
    }

    private static Collection<PropertyDescriptor> sortProperties(Collection<PropertyDescriptor> properties) {
        List<PropertyDescriptor> result = newArrayList(properties);
        Collections.sort(result, new Comparator<PropertyDescriptor>() {
            @Override
            public int compare(PropertyDescriptor o1, PropertyDescriptor o2) {
                int result = o1.isRequired() == o2.isRequired() ? 0 : (o1.isRequired() ? -1 : 1);
                if (result == 0) {
                    result = o1.getName().compareTo(o2.getName());
                }
                return result;
            }
        });
        return result;
    }

    private static Collection<MethodDescriptor> sortMethods(Collection<MethodDescriptor> methods) {
        List<MethodDescriptor> result = newArrayList(methods);
        Collections.sort(result, new Comparator<MethodDescriptor>() {
            @Override
            public int compare(MethodDescriptor o1, MethodDescriptor o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });
        return result;
    }

    enum CiPredicates implements Predicate<Descriptor> {
        DEPLOYABLE {
            public boolean apply(Descriptor type) {
                return type.isAssignableTo(Deployable.class);
            }
        },
        DEPLOYED {
            public boolean apply(Descriptor type) {
                return type.isAssignableTo(Deployed.class);
            }
        },
        CONTAINER {
            public boolean apply(Descriptor type) {
                return type.isAssignableTo(Container.class);
            }
        };

        static Predicate<Descriptor> otherCIs() {
            return not(or(values()));
        }
    }

    enum PropertyPredicates implements Predicate<PropertyDescriptor> {
        PUBLIC {
            public boolean apply(PropertyDescriptor property) {
                return !property.isHidden() && !isParent(property) && !isChild(property);
            }
        },
        PARENT {
            public boolean apply(PropertyDescriptor property) {
                return isParent(property) && !property.isHidden();
            }
        },
        CHILDREN {
            public boolean apply(PropertyDescriptor property) {
                return isChild(property) && !property.isHidden();
            }
        },
        HIDDEN {
            public boolean apply(PropertyDescriptor property) {
                return property.isHidden();
            }
        };

        static boolean isParent(PropertyDescriptor ci) {
            return ci.isAsContainment() && ci.getKind() == PropertyKind.CI;
        }

        static boolean isChild(PropertyDescriptor ci) {
            return ci.isAsContainment() && ci.getKind() != PropertyKind.CI;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(CiReferenceGenerator.class);
}
